<?php
/*
Plugin Name: WPC Product Timer for WooCommerce
Plugin URI: https://wpclever.net/
Description: WPC Product Timer helps you add many actions for the product based on the conditionals of the time.
Version: 2.1.3
Author: WPClever.net
Author URI: https://wpclever.net
Text Domain: woopt
Domain Path: /languages/
Requires at least: 4.0
Tested up to: 5.5.1
WC requires at least: 3.0
WC tested up to: 4.6.1
*/

defined( 'ABSPATH' ) || exit;

! defined( 'WOOPT_VERSION' ) && define( 'WOOPT_VERSION', '2.1.3' );
! defined( 'WOOPT_URI' ) && define( 'WOOPT_URI', plugin_dir_url( __FILE__ ) );
! defined( 'WOOPT_DOCS' ) && define( 'WOOPT_DOCS', 'http://doc.wpclever.net/woopt/' );
! defined( 'WOOPT_SUPPORT' ) && define( 'WOOPT_SUPPORT', 'https://wpclever.net/support?utm_source=support&utm_medium=woopt&utm_campaign=wporg' );
! defined( 'WOOPT_REVIEWS' ) && define( 'WOOPT_REVIEWS', 'https://wordpress.org/support/plugin/woo-product-timer/reviews/?filter=5' );
! defined( 'WOOPT_CHANGELOG' ) && define( 'WOOPT_CHANGELOG', 'https://wordpress.org/plugins/woo-product-timer/#developers' );
! defined( 'WOOPT_DISCUSSION' ) && define( 'WOOPT_DISCUSSION', 'https://wordpress.org/support/plugin/woo-product-timer' );
! defined( 'WPC_URI' ) && define( 'WPC_URI', WOOPT_URI );

include 'includes/wpc-dashboard.php';
include 'includes/wpc-menu.php';
include 'includes/wpc-kit.php';

if ( ! function_exists( 'woopt_init' ) ) {
	add_action( 'plugins_loaded', 'woopt_init', 11 );

	function woopt_init() {
		// load text-domain
		load_plugin_textdomain( 'woopt', false, basename( __DIR__ ) . '/languages/' );

		if ( ! function_exists( 'WC' ) || ! version_compare( WC()->version, '3.0.0', '>=' ) ) {
			add_action( 'admin_notices', 'woopt_notice_wc' );

			return;
		}

		if ( ! class_exists( 'WPCleverWoopt' ) && class_exists( 'WC_Product' ) ) {
			class WPCleverWoopt {
				function __construct() {
					// Menu
					add_action( 'admin_menu', array( $this, 'woopt_admin_menu' ) );

					// Enqueue backend scripts
					add_action( 'admin_enqueue_scripts', array( $this, 'woopt_admin_enqueue_scripts' ) );

					// Product data tabs
					add_filter( 'woocommerce_product_data_tabs', array( $this, 'woopt_product_data_tabs' ), 10, 1 );

					// Product data panels
					add_action( 'woocommerce_product_data_panels', array( $this, 'woopt_product_data_panels' ) );
					add_action( 'woocommerce_process_product_meta', array( $this, 'woopt_save_option_field' ) );

					// Add settings link
					add_filter( 'plugin_action_links', array( $this, 'woopt_action_links' ), 10, 2 );
					add_filter( 'plugin_row_meta', array( $this, 'woopt_row_meta' ), 10, 2 );

					// AJAX
					add_action( 'wp_ajax_woopt_add_conditional', array( $this, 'woopt_add_conditional' ) );

					// Product actions
					add_filter( 'woocommerce_product_is_in_stock', array( $this, 'woopt_is_in_stock' ), 99, 2 );
					add_filter( 'woocommerce_product_is_visible', array( $this, 'woopt_is_visible' ), 99, 2 );
					add_filter( 'woocommerce_is_purchasable', array( $this, 'woopt_is_purchasable' ), 99, 2 );
					add_filter( 'woocommerce_is_sold_individually', array( $this, 'woopt_sold_individually' ), 99, 2 );
					add_filter( 'woocommerce_product_get_featured', array( $this, 'woopt_is_featured' ), 99, 2 );
					add_filter( 'woocommerce_product_get_regular_price', array(
						$this,
						'woopt_get_regular_price'
					), 99, 2 );
					add_filter( 'woocommerce_product_get_sale_price', array( $this, 'woopt_get_sale_price' ), 99, 2 );
					add_filter( 'woocommerce_product_get_price', array( $this, 'woopt_get_price' ), 99, 2 );

					// Variation
					add_filter( 'woocommerce_product_variation_get_regular_price', array(
						$this,
						'woopt_get_regular_price'
					), 99, 2 );
					add_filter( 'woocommerce_product_variation_get_sale_price', array(
						$this,
						'woopt_get_sale_price'
					), 99, 2 );
					add_filter( 'woocommerce_product_variation_get_price', array( $this, 'woopt_get_price' ), 99, 2 );

					// Variations
					add_filter( 'woocommerce_variation_prices_regular_price', array(
						$this,
						'woopt_get_regular_price'
					), 99, 2 );
					add_filter( 'woocommerce_variation_prices_sale_price', array(
						$this,
						'woopt_get_sale_price'
					), 99, 2 );
					add_filter( 'woocommerce_variation_prices_price', array( $this, 'woopt_get_price' ), 99, 2 );

					// Product columns
					add_filter( 'manage_edit-product_columns', array( $this, 'woopt_product_columns' ), 10 );
					add_action( 'manage_product_posts_custom_column', array(
						$this,
						'woopt_product_posts_custom_column'
					), 10, 2 );

					// Product class
					add_filter( 'woocommerce_post_class', array( $this, 'woopt_post_class' ), 99, 2 );

					// Export
					add_filter( 'woocommerce_product_export_column_names', array( $this, 'woopt_add_export_column' ) );
					add_filter( 'woocommerce_product_export_product_default_columns', array(
						$this,
						'woopt_add_export_column'
					) );
					add_filter( 'woocommerce_product_export_product_column_woopt_actions', array(
						$this,
						'woopt_add_export_data'
					), 10, 2 );

					// Import
					add_filter( 'woocommerce_csv_product_import_mapping_options', array(
						$this,
						'woopt_add_column_to_importer'
					) );
					add_filter( 'woocommerce_csv_product_import_mapping_default_columns', array(
						$this,
						'woopt_add_column_to_mapping_screen'
					) );
					add_filter( 'woocommerce_product_import_pre_insert_product_object', array(
						$this,
						'woopt_process_import'
					), 10, 2 );
				}

				function woopt_post_class( $classes, $product ) {
					if ( $product && $product->is_type( 'variation' ) && $product->get_parent_id() ) {
						$product_id = $product->get_parent_id();
					} else {
						$product_id = $product->get_id();
					}

					$actions = get_post_meta( $product_id, 'woopt_actions', true );

					if ( is_array( $actions ) && ( count( $actions ) > 0 ) ) {
						foreach ( $actions as $action ) {
							$action_data = $this->woopt_action_data( $action );

							if ( $this->woopt_check_conditional( $action_data['conditional'] ) ) {
								$classes[] = 'woopt';
								$classes[] = 'woopt_' . $action_data['action'];
							}
						}
					}

					return $classes;
				}

				function woopt_get_regular_price( $regular_price, $product ) {
					if ( $product->is_type( 'variation' ) && $product->get_parent_id() ) {
						$product_id = $product->get_parent_id();
					} else {
						$product_id = $product->get_id();
					}

					$actions = get_post_meta( $product_id, 'woopt_actions', true );

					if ( is_array( $actions ) && ( count( $actions ) > 0 ) ) {
						foreach ( $actions as $action ) {
							$action_data        = $this->woopt_action_data( $action );
							$action_key         = $action_data['action'];
							$action_price       = $action_data['value'];
							$action_conditional = $action_data['conditional'];

							if ( ( $action_key === 'set_regularprice' ) && $this->woopt_check_conditional( $action_conditional ) ) {
								if ( strpos( $action_price, '%' ) !== false ) {
									$percentage   = (float) preg_replace( '/[^0-9.]/', '', $action_price );
									$action_price = (float) $regular_price * $percentage / 100;
								}

								$regular_price = $action_price;
							}
						}
					}

					return $regular_price;
				}

				function woopt_get_sale_price( $sale_price, $product ) {
					if ( $product->is_type( 'variation' ) && $product->get_parent_id() ) {
						$product_id = $product->get_parent_id();
					} else {
						$product_id = $product->get_id();
					}

					$actions = get_post_meta( $product_id, 'woopt_actions', true );

					if ( is_array( $actions ) && ( count( $actions ) > 0 ) ) {
						foreach ( $actions as $action ) {
							$action_data        = $this->woopt_action_data( $action );
							$action_key         = $action_data['action'];
							$action_price       = $action_data['value'];
							$action_conditional = $action_data['conditional'];

							if ( ( $action_key === 'set_saleprice' ) && $this->woopt_check_conditional( $action_conditional ) ) {
								if ( strpos( $action_price, '%' ) !== false ) {
									$percentage   = (float) preg_replace( '/[^0-9.]/', '', $action_price );
									$action_price = (float) $sale_price * $percentage / 100;
								}

								$sale_price = $action_price;
							}
						}
					}

					return $sale_price;
				}

				function woopt_get_price( $price, $product ) {
					if ( $product->is_on_sale() ) {
						return $this->woopt_get_sale_price( $price, $product );
					}

					return $this->woopt_get_regular_price( $price, $product );
				}

				function woopt_is_in_stock( $instock, $product ) {
					if ( $product->is_type( 'variation' ) && $product->get_parent_id() ) {
						$product_id = $product->get_parent_id();
					} else {
						$product_id = $product->get_id();
					}

					$actions = get_post_meta( $product_id, 'woopt_actions', true );

					if ( is_array( $actions ) && ( count( $actions ) > 0 ) ) {
						foreach ( $actions as $action ) {
							$action_data        = $this->woopt_action_data( $action );
							$action_key         = $action_data['action'];
							$action_conditional = $action_data['conditional'];

							if ( ( $action_key === 'set_instock' ) && $this->woopt_check_conditional( $action_conditional ) ) {
								$instock = true;
							}

							if ( ( $action_key === 'set_outofstock' ) && $this->woopt_check_conditional( $action_conditional ) ) {
								$instock = false;
							}
						}
					}

					return $instock;
				}

				function woopt_is_visible( $visible, $product_id ) {
					$actions = get_post_meta( $product_id, 'woopt_actions', true );

					if ( is_array( $actions ) && ( count( $actions ) > 0 ) ) {
						foreach ( $actions as $action ) {
							$action_data        = $this->woopt_action_data( $action );
							$action_key         = $action_data['action'];
							$action_conditional = $action_data['conditional'];

							if ( ( $action_key === 'set_visible' ) && $this->woopt_check_conditional( $action_conditional ) ) {
								$visible = true;
							}

							if ( ( $action_key === 'set_hidden' ) && $this->woopt_check_conditional( $action_conditional ) ) {
								$visible = false;
							}
						}
					}

					return $visible;
				}

				function woopt_is_featured( $featured, $product ) {
					if ( $product->is_type( 'variation' ) && $product->get_parent_id() ) {
						$product_id = $product->get_parent_id();
					} else {
						$product_id = $product->get_id();
					}

					$actions = get_post_meta( $product_id, 'woopt_actions', true );

					if ( is_array( $actions ) && ( count( $actions ) > 0 ) ) {
						foreach ( $actions as $action ) {
							$action_data        = $this->woopt_action_data( $action );
							$action_key         = $action_data['action'];
							$action_conditional = $action_data['conditional'];

							if ( ( $action_key === 'set_featured' ) && $this->woopt_check_conditional( $action_conditional ) ) {
								$featured = true;
							}

							if ( ( $action_key === 'set_unfeatured' ) && $this->woopt_check_conditional( $action_conditional ) ) {
								$featured = false;
							}
						}
					}

					return $featured;
				}

				function woopt_is_purchasable( $purchasable, $product ) {
					if ( $product->is_type( 'variation' ) && $product->get_parent_id() ) {
						$product_id = $product->get_parent_id();
					} else {
						$product_id = $product->get_id();
					}

					$actions = get_post_meta( $product_id, 'woopt_actions', true );

					if ( is_array( $actions ) && ( count( $actions ) > 0 ) ) {
						foreach ( $actions as $action ) {
							$action_data        = $this->woopt_action_data( $action );
							$action_key         = $action_data['action'];
							$action_conditional = $action_data['conditional'];

							if ( ( $action_key === 'set_purchasable' ) && $this->woopt_check_conditional( $action_conditional ) ) {
								$purchasable = true;
							}

							if ( ( $action_key === 'set_unpurchasable' ) && $this->woopt_check_conditional( $action_conditional ) ) {
								$purchasable = false;
							}
						}
					}

					return $purchasable;
				}

				function woopt_sold_individually( $sold_individually, $product ) {
					if ( $product->is_type( 'variation' ) && $product->get_parent_id() ) {
						$product_id = $product->get_parent_id();
					} else {
						$product_id = $product->get_id();
					}

					$actions = get_post_meta( $product_id, 'woopt_actions', true );

					if ( is_array( $actions ) && ( count( $actions ) > 0 ) ) {
						foreach ( $actions as $action ) {
							$action_data        = $this->woopt_action_data( $action );
							$action_key         = $action_data['action'];
							$action_conditional = $action_data['conditional'];

							if ( ( $action_key === 'enable_sold_individually' ) && $this->woopt_check_conditional( $action_conditional ) ) {
								$sold_individually = true;
							}

							if ( ( $action_key === 'disable_sold_individually' ) && $this->woopt_check_conditional( $action_conditional ) ) {
								$sold_individually = false;
							}
						}
					}

					return $sold_individually;
				}

				function woopt_admin_menu() {
					add_submenu_page( 'wpclever', esc_html__( 'WPC Product Timer', 'woopt' ), esc_html__( 'Product Timer', 'woopt' ), 'manage_options', 'wpclever-woopt', array(
						&$this,
						'woopt_admin_menu_content'
					) );
				}

				function woopt_admin_menu_content() {
					$active_tab = isset( $_GET['tab'] ) ? $_GET['tab'] : 'how';
					?>
                    <div class="wpclever_settings_page wrap">
                        <h1 class="wpclever_settings_page_title"><?php echo esc_html__( 'WPC Product Timer', 'woopt' ) . ' ' . WOOPT_VERSION; ?></h1>
                        <div class="wpclever_settings_page_desc about-text">
                            <p>
								<?php printf( esc_html__( 'Thank you for using our plugin! If you are satisfied, please reward it a full five-star %s rating.', 'woopt' ), '<span style="color:#ffb900">&#9733;&#9733;&#9733;&#9733;&#9733;</span>' ); ?>
                                <br/>
                                <a href="<?php echo esc_url( WOOPT_REVIEWS ); ?>"
                                   target="_blank"><?php esc_html_e( 'Reviews', 'woopt' ); ?></a> | <a
                                        href="<?php echo esc_url( WOOPT_CHANGELOG ); ?>"
                                        target="_blank"><?php esc_html_e( 'Changelog', 'woopt' ); ?></a>
                                | <a href="<?php echo esc_url( WOOPT_DISCUSSION ); ?>"
                                     target="_blank"><?php esc_html_e( 'Discussion', 'woopt' ); ?></a>
                            </p>
                        </div>
                        <div class="wpclever_settings_page_nav">
                            <h2 class="nav-tab-wrapper">
                                <a href="<?php echo admin_url( 'admin.php?page=wpclever-woopt&tab=how' ); ?>"
                                   class="<?php echo $active_tab === 'how' ? 'nav-tab nav-tab-active' : 'nav-tab'; ?>">
									<?php esc_html_e( 'How to use?', 'woopt' ); ?>
                                </a>
                                <a href="<?php echo esc_url( WOOPT_DOCS ); ?>" class="nav-tab" target="_blank">
									<?php esc_html_e( 'Docs', 'woopt' ); ?>
                                </a>
                                <a href="<?php echo admin_url( 'admin.php?page=wpclever-woopt&tab=premium' ); ?>"
                                   class="<?php echo $active_tab === 'premium' ? 'nav-tab nav-tab-active' : 'nav-tab'; ?>"
                                   style="color: #c9356e">
									<?php esc_html_e( 'Premium Version', 'woopt' ); ?>
                                </a>
                                <a href="<?php echo esc_url( admin_url( 'admin.php?page=wpclever-kit' ) ); ?>"
                                   class="nav-tab">
									<?php esc_html_e( 'Essential Kit', 'woopt' ); ?>
                                </a>
                            </h2>
                        </div>
                        <div class="wpclever_settings_page_content">
							<?php if ( $active_tab === 'how' ) { ?>
                                <div class="wpclever_settings_page_content_text">
                                    <p>
										<?php esc_html_e( 'When adding/editing the product you can choose the Timer tab then add action & time conditional.', 'woopt' ); ?>
                                    </p>
                                    <p>
                                        <img src="<?php echo esc_url( WOOPT_URI ); ?>assets/images/how-01.jpg"/>
                                    </p>
                                </div>
							<?php } elseif ( $active_tab === 'premium' ) { ?>
                                <div class="wpclever_settings_page_content_text">
                                    <p>Get the Premium Version just $29! <a
                                                href="https://wpclever.net/downloads/woocommerce-product-timer?utm_source=pro&utm_medium=woopt&utm_campaign=wporg"
                                                target="_blank">https://wpclever.net/downloads/woocommerce-product-timer</a>
                                    </p>
                                    <p><strong>Extra features for Premium Version:</strong></p>
                                    <ul style="margin-bottom: 0">
                                        <li>- Add multiple actions for one product.</li>
                                        <li>- Get the lifetime update & premium support.</li>
                                    </ul>
                                </div>
							<?php } ?>
                        </div>
                    </div>
					<?php
				}

				function woopt_admin_enqueue_scripts() {
					wp_enqueue_style( 'wooptdpk', WOOPT_URI . 'assets/libs/wooptdpk/css/datepicker.css' );
					wp_enqueue_script( 'wooptdpk', WOOPT_URI . 'assets/libs/wooptdpk/js/datepicker.js', array( 'jquery' ), WOOPT_VERSION, true );
					wp_enqueue_style( 'woopt-backend', WOOPT_URI . 'assets/css/backend.css' );
					wp_enqueue_script( 'woopt-backend', WOOPT_URI . 'assets/js/backend.js', array( 'jquery' ), WOOPT_VERSION, true );
					wp_localize_script( 'woopt-backend', 'woopt_vars', array(
							'woopt_nonce' => wp_create_nonce( 'woopt_nonce' )
						)
					);
				}

				function woopt_action_links( $links, $file ) {
					static $plugin;

					if ( ! isset( $plugin ) ) {
						$plugin = plugin_basename( __FILE__ );
					}

					if ( $plugin === $file ) {
						$settings_link    = '<a href="' . admin_url( 'admin.php?page=wpclever-woopt&tab=how' ) . '">' . esc_html__( 'How to use?', 'woopt' ) . '</a>';
						$links['premium'] = '<a href="' . admin_url( 'admin.php?page=wpclever-woopt&tab=premium' ) . '">' . esc_html__( 'Premium Version', 'woopt' ) . '</a>';
						array_unshift( $links, $settings_link );
					}

					return (array) $links;
				}

				function woopt_row_meta( $links, $file ) {
					static $plugin;

					if ( ! isset( $plugin ) ) {
						$plugin = plugin_basename( __FILE__ );
					}

					if ( $plugin === $file ) {
						$row_meta = array(
							'docs'    => '<a href="' . esc_url( WOOPT_DOCS ) . '" target="_blank">' . esc_html__( 'Docs', 'woopt' ) . '</a>',
							'support' => '<a href="' . esc_url( WOOPT_SUPPORT ) . '" target="_blank">' . esc_html__( 'Support', 'woopt' ) . '</a>',
						);

						return array_merge( $links, $row_meta );
					}

					return (array) $links;
				}

				function woopt_product_data_tabs( $tabs ) {
					$tabs['woopt'] = array(
						'label'  => esc_html__( 'Timer', 'woopt' ),
						'target' => 'woopt_settings',
					);

					return $tabs;
				}

				function woopt_product_data_panels() {
					global $post;
					$post_id = $post->ID;
					$actions = get_post_meta( $post_id, 'woopt_actions', true );
					?>
                    <div id='woopt_settings' class='panel woocommerce_options_panel woopt_table'>
                        <div class="woopt_current_time">
							<?php esc_html_e( 'Current time', 'woopt' ); ?>
                            <code><?php echo current_time( 'l' ); ?></code>
                            <code><?php echo current_time( 'm/d/Y' ); ?></code>
                            <code><?php echo current_time( 'h:i a' ); ?></code>
                            <code><?php echo esc_html__( 'Week No.', 'woopt' ) . ' ' . current_time( 'W' ); ?></code>
                            <a
                                    href="<?php echo admin_url( 'options-general.php' ); ?>"
                                    target="_blank"><?php esc_html_e( 'Date/time settings', 'woopt' ); ?></a>
                        </div>
                        <div class="woopt_tables">
							<?php
							if ( is_array( $actions ) && ( count( $actions ) > 0 ) ) {
								foreach ( $actions as $action ) {
									$this->woopt_action( $action );
								}
							} else {
								$this->woopt_action();
							}
							?>
                        </div>
                        <div class="woopt_add_action">
                            <a href="https://wpclever.net/downloads/woocommerce-product-timer?utm_source=pro&utm_medium=woopt&utm_campaign=wporg"
                               target="_blank"
                               onclick="return confirm('This feature only available in Premium Version!\nBuy it now? Just $29')">
								<?php esc_html_e( '+ Add action', 'woopt' ); ?>
                            </a>
                        </div>
                    </div>
					<?php
				}

				function woopt_action_data( $action ) {
					$action_arr = explode( '|', $action );

					return array(
						'action'      => isset( $action_arr[0] ) ? $action_arr[0] : '',
						'value'       => isset( $action_arr[1] ) ? $action_arr[1] : '100%',
						'conditional' => isset( $action_arr[2] ) ? $action_arr[2] : '',
					);
				}

				function woopt_action( $action = null ) {
					$action_data = $this->woopt_action_data( $action );
					$action      = $action_data['action'];
					$price       = $action_data['value'];
					$conditional = $action_data['conditional'];
					?>
                    <div class="woopt_table">
                        <span class="woopt_action_remove">&times;</span>
                        <div class="woopt_tr">
                            <div class="woopt_th"><?php esc_html_e( 'Action', 'woopt' ); ?></div>
                            <div class="woopt_td woopt_action_td">
                                <input class="woopt_action_val" type="hidden" name="woopt_action_val[]" value=""/>
                                <span>
							<select class="woopt_action">
								<option value=""><?php esc_html_e( 'Choose action', 'woopt' ); ?></option>
								<option value="set_instock" <?php echo( $action === 'set_instock' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Set in stock', 'woopt' ); ?>
								</option>
								<option value="set_outofstock" <?php echo( $action === 'set_outofstock' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Set out of stock', 'woopt' ); ?>
								</option>
								<option value="set_visible" <?php echo( $action === 'set_visible' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Set visible', 'woopt' ); ?>
								</option>
								<option value="set_hidden" <?php echo( $action === 'set_hidden' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Set hidden', 'woopt' ); ?>
								</option>
                                <option value="set_featured" <?php echo( $action === 'set_featured' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Set featured', 'woopt' ); ?>
								</option>
                                <option value="set_unfeatured" <?php echo( $action === 'set_unfeatured' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Set unfeatured', 'woopt' ); ?>
								</option>
								<option value="set_purchasable" <?php echo( $action === 'set_purchasable' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Set purchasable', 'woopt' ); ?>
								</option>
								<option value="set_unpurchasable" <?php echo( $action === 'set_unpurchasable' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Set unpurchasable', 'woopt' ); ?>
								</option>
								<option value="set_regularprice"
                                        data-show="price" <?php echo( $action === 'set_regularprice' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Set regular price', 'woopt' ); ?>
								</option>
								<option value="set_saleprice"
                                        data-show="price" <?php echo( $action === 'set_saleprice' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Set sale price', 'woopt' ); ?>
								</option>
                                <option value="enable_sold_individually"  <?php echo( $action === 'enable_sold_individually' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Enable sold individually', 'woopt' ); ?>
								</option>
                                <option value="disable_sold_individually"  <?php echo( $action === 'disable_sold_individually' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Disable sold individually', 'woopt' ); ?>
								</option>
							</select>
						</span>
                                <span class="woopt_hide woopt_show_if_price">
							<input class="woopt_price"
                                   value="<?php echo $price; ?>"
                                   type="text" placeholder="price or percentage" style="width: 150px"/>
						</span>
                            </div>
                        </div>
                        <div class="woopt_tr">
                            <div class="woopt_th"><?php esc_html_e( 'Time conditional', 'woopt' ); ?></div>
                            <div class="woopt_td">
                                <div class="woopt_conditionals">
									<?php $this->woopt_conditional( $conditional ); ?>
                                </div>
                                <div class="woopt_add_conditional">
                                    <a href="#" class="woopt_new_conditional">+ Add time</a>
                                </div>
                            </div>
                        </div>
                    </div>
					<?php
				}

				function woopt_conditional( $conditional = null ) {
					$conditional_arr = explode( '&', $conditional );

					if ( is_array( $conditional_arr ) && ( count( $conditional_arr ) > 0 ) ) {
						foreach ( $conditional_arr as $conditional_item ) {
							$conditional_item_arr       = explode( '>', $conditional_item );
							$conditional_item_key       = isset( $conditional_item_arr[0] ) ? $conditional_item_arr[0] : '';
							$conditional_item_val       = trim( isset( $conditional_item_arr[1] ) ? $conditional_item_arr[1] : '' );
							$conditional_item_time_from = '';
							$conditional_item_time_to   = '';

							if ( $conditional_item_key === 'time_range' ) {
								$conditional_item_time      = explode( '-', $conditional_item_val );
								$conditional_item_time_from = trim( isset( $conditional_item_time[0] ) ? $conditional_item_time[0] : '' );
								$conditional_item_time_to   = trim( isset( $conditional_item_time[1] ) ? $conditional_item_time[1] : '' );
							}
							?>
                            <div class="woopt_conditional_item">
                                <span class="woopt_conditional_remove">&times;</span>
                                <span>
							<select class="woopt_conditional">
								<option value=""><?php esc_html_e( 'Choose the time', 'woopt' ); ?></option>
								<option value="date_on"
                                        data-show="date" <?php echo( $conditional_item_key === 'date_on' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'On the date', 'woopt' ); ?>
								</option>
                                <option value="date_time_before"
                                        data-show="date_time" <?php echo( $conditional_item_key === 'date_time_before' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Before date & time', 'woopt' ); ?>
								</option>
								<option value="date_time_after"
                                        data-show="date_time" <?php echo( $conditional_item_key === 'date_time_after' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'After date & time', 'woopt' ); ?>
								</option>
								<option value="date_before"
                                        data-show="date" <?php echo( $conditional_item_key === 'date_before' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Before date', 'woopt' ); ?>
								</option>
								<option value="date_after"
                                        data-show="date" <?php echo( $conditional_item_key === 'date_after' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'After date', 'woopt' ); ?>
								</option>
								<option value="date_multi"
                                        data-show="date_multi" <?php echo( $conditional_item_key === 'date_multi' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Multiple dates', 'woopt' ); ?>
								</option>
								<option value="date_range"
                                        data-show="date_range" <?php echo( $conditional_item_key === 'date_range' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Date range', 'woopt' ); ?>
								</option>
								<option value="date_even"
                                        data-show="none" <?php echo( $conditional_item_key === 'date_even' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'All even dates', 'woopt' ); ?>
								</option>
								<option value="date_odd"
                                        data-show="none" <?php echo( $conditional_item_key === 'date_odd' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'All odd dates', 'woopt' ); ?>
								</option>
								<option value="time_range"
                                        data-show="time_range" <?php echo( $conditional_item_key === 'time_range' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Daily time range', 'woopt' ); ?>
								</option>
                                <option value="time_before"
                                        data-show="time" <?php echo( $conditional_item_key === 'time_before' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Daily before time', 'woopt' ); ?>
								</option>
								<option value="time_after"
                                        data-show="time" <?php echo( $conditional_item_key === 'time_after' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Daily after time', 'woopt' ); ?>
								</option>
                                <option value="weekly_every"
                                        data-show="weekday" <?php echo( $conditional_item_key === 'weekly_every' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Weekly on every', 'woopt' ); ?>
								</option>
                                <option value="week_no"
                                        data-show="weekno" <?php echo( $conditional_item_key === 'week_no' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'On week No.', 'woopt' ); ?>
								</option>
                                <option value="every_day"
                                        data-show="none" <?php echo( $conditional_item_key === 'every_day' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Everyday', 'woopt' ); ?>
								</option>
							</select>
						</span>
                                <span class="woopt_hide woopt_show_if_date_time">
							<input value="<?php echo $conditional_item_val; ?>"
                                   class="woopt_date_time woopt_date_time_input woopt_picker"
                                   type="text"
                                   readonly="readonly" style="width: 300px"/>
						</span>
                                <span class="woopt_hide woopt_show_if_date">
							<input value="<?php echo $conditional_item_val; ?>"
                                   class="woopt_date woopt_date_input woopt_picker"
                                   type="text"
                                   readonly="readonly" style="width: 300px"/>
						</span>
                                <span class="woopt_hide woopt_show_if_date_range">
							<input value="<?php echo $conditional_item_val; ?>"
                                   class="woopt_date_range woopt_date_input woopt_picker"
                                   type="text"
                                   readonly="readonly" style="width: 300px"/>
						</span>
                                <span class="woopt_hide woopt_show_if_date_multi">
							<input value="<?php echo $conditional_item_val; ?>"
                                   class="woopt_date_multi woopt_date_input woopt_picker"
                                   type="text"
                                   readonly="readonly" style="width: 300px"/>
						</span>
                                <span class="woopt_hide woopt_show_if_time_range">
							<input value="<?php echo $conditional_item_time_from; ?>"
                                   class="woopt_time woopt_time_start woopt_time_input woopt_picker" type="text"
                                   readonly="readonly" style="width: 300px" placeholder="from"/>
							<input value="<?php echo $conditional_item_time_to; ?>"
                                   class="woopt_time woopt_time_end woopt_time_input woopt_picker"
                                   type="text"
                                   readonly="readonly"
                                   style="width: 300px" placeholder="to"/>
						</span>
                                <span class="woopt_hide woopt_show_if_time">
							<input value="<?php echo $conditional_item_val; ?>"
                                   class="woopt_time woopt_time_on woopt_time_input woopt_picker"
                                   type="text"
                                   readonly="readonly" style="width: 300px"/>
						</span>
                                <span class="woopt_hide woopt_show_if_weekday">
							<select class="woopt_weekday">
                                <option value="mon" <?php echo( $conditional_item_val === 'mon' ? 'selected' : '' ); ?>><?php esc_html_e( 'Monday', 'woopt' ); ?></option>
                                <option value="tue" <?php echo( $conditional_item_val === 'tue' ? 'selected' : '' ); ?>><?php esc_html_e( 'Tuesday', 'woopt' ); ?></option>
                                <option value="wed" <?php echo( $conditional_item_val === 'wed' ? 'selected' : '' ); ?>><?php esc_html_e( 'Wednesday', 'woopt' ); ?></option>
                                <option value="thu" <?php echo( $conditional_item_val === 'thu' ? 'selected' : '' ); ?>><?php esc_html_e( 'Thursday', 'woopt' ); ?></option>
                                <option value="fri" <?php echo( $conditional_item_val === 'fri' ? 'selected' : '' ); ?>><?php esc_html_e( 'Friday', 'woopt' ); ?></option>
                                <option value="sat" <?php echo( $conditional_item_val === 'sat' ? 'selected' : '' ); ?>><?php esc_html_e( 'Saturday', 'woopt' ); ?></option>
                                <option value="sun" <?php echo( $conditional_item_val === 'sun' ? 'selected' : '' ); ?>><?php esc_html_e( 'Sunday', 'woopt' ); ?></option>
                            </select>
						</span>
                                <span class="woopt_hide woopt_show_if_weekno">
							<select class="woopt_weekno">
                                <?php
                                for ( $i = 1; $i < 54; $i ++ ) {
	                                echo '<option value="' . esc_attr( $i ) . '" ' . ( (int) $conditional_item_val === $i ? 'selected' : '' ) . '>' . $i . '</option>';
                                }
                                ?>
                            </select>
						</span>
                            </div>
							<?php
						}
					}
				}

				function woopt_add_conditional() {
					if ( ! isset( $_POST['woopt_nonce'] ) || ! wp_verify_nonce( $_POST['woopt_nonce'], 'woopt_nonce' ) ) {
						die( 'Permissions check failed' );
					}

					$this->woopt_conditional();
					die();
				}

				function woopt_save_option_field( $post_id ) {
					if ( isset( $_POST['woopt_action_val'] ) && is_array( $_POST['woopt_action_val'] ) ) {
						update_post_meta( $post_id, 'woopt_actions', $_POST['woopt_action_val'] );
					} else {
						delete_post_meta( $post_id, 'woopt_actions' );
					}
				}

				function woopt_check_conditional( $conditionals ) {
					$condition        = true;
					$conditionals_arr = explode( '&', $conditionals );

					if ( count( $conditionals_arr ) > 0 ) {
						foreach ( $conditionals_arr as $conditional ) {
							$condition_item  = false;
							$conditional_arr = explode( '>', $conditional );

							if ( count( $conditional_arr ) > 0 ) {
								$conditional_key   = isset( $conditional_arr[0] ) ? $conditional_arr[0] : '';
								$conditional_value = isset( $conditional_arr[1] ) ? $conditional_arr[1] : '';

								if ( $conditional_value !== '' ) {
									switch ( $conditional_key ) {
										case 'date_range':
											$date_range_arr = explode( '-', $conditional_value );

											if ( count( $date_range_arr ) === 2 ) {
												$date_range_start = trim( $date_range_arr[0] );
												$date_range_end   = trim( $date_range_arr[1] );
												$current_time     = strtotime( current_time( 'm/d/Y' ) );

												if ( $current_time >= strtotime( $date_range_start ) && $current_time <= strtotime( $date_range_end ) ) {
													$condition_item = true;
												}
											} elseif ( count( $date_range_arr ) === 1 ) {
												$date_range_start = trim( $date_range_arr[0] );

												if ( strtotime( current_time( 'm/d/Y' ) ) === strtotime( $date_range_start ) ) {
													$condition_item = true;
												}
											}

											break;
										case 'date_multi':
											$multiple_dates_arr = explode( ', ', $conditional_value );

											if ( in_array( current_time( 'm/d/Y' ), $multiple_dates_arr, false ) ) {
												$condition_item = true;
											}

											break;
										case 'date_even':
											if ( (int) current_time( 'd' ) % 2 === 0 ) {
												$condition_item = true;
											}

											break;
										case 'date_odd':
											if ( (int) current_time( 'd' ) % 2 !== 0 ) {
												$condition_item = true;
											}

											break;
										case 'date_on':
											if ( strtotime( current_time( 'm/d/Y' ) ) === strtotime( $conditional_value ) ) {
												$condition_item = true;
											}

											break;
										case 'date_before':
											if ( strtotime( current_time( 'm/d/Y' ) ) < strtotime( $conditional_value ) ) {
												$condition_item = true;
											}

											break;
										case 'date_after':
											if ( strtotime( current_time( 'm/d/Y' ) ) > strtotime( $conditional_value ) ) {
												$condition_item = true;
											}

											break;
										case 'date_time_before':
											if ( current_time( 'timestamp' ) < strtotime( $conditional_value ) ) {
												$condition_item = true;
											}

											break;
										case 'date_time_after':
											if ( current_time( 'timestamp' ) > strtotime( $conditional_value ) ) {
												$condition_item = true;
											}

											break;
										case 'time_range':
											$time_range_arr = explode( '-', $conditional_value );

											if ( count( $time_range_arr ) === 2 ) {
												$current_time     = current_time( 'timestamp' );
												$current_date     = current_time( 'm/d/Y' );
												$time_range_start = $current_date . ' ' . trim( $time_range_arr[0] );
												$time_range_end   = $current_date . ' ' . trim( $time_range_arr[1] );

												if ( $current_time >= strtotime( $time_range_start ) && $current_time <= strtotime( $time_range_end ) ) {
													$condition_item = true;
												}
											}

											break;
										case 'time_before':
											$current_date = current_time( 'm/d/Y' );

											if ( current_time( 'timestamp' ) < strtotime( $current_date . ' ' . $conditional_value ) ) {
												$condition_item = true;
											}

											break;
										case 'time_after':
											$current_date = current_time( 'm/d/Y' );

											if ( current_time( 'timestamp' ) > strtotime( $current_date . ' ' . $conditional_value ) ) {
												$condition_item = true;
											}

											break;
										case 'weekly_every':
											if ( strtolower( current_time( 'D' ) ) === $conditional_value ) {
												$condition_item = true;
											}

											break;
										case 'week_no':
											if ( (int) current_time( 'W' ) === (int) $conditional_value ) {
												$condition_item = true;
											}

											break;
										case 'every_day':
											$condition_item = true;

											break;
									}
								}
							}

							$condition &= $condition_item;
						}
					}

					return $condition;
				}

				function woopt_product_columns( $columns ) {
					$columns['woopt'] = esc_html__( 'Timer', 'woopt' );

					return $columns;
				}

				function woopt_product_posts_custom_column( $column, $postid ) {
					if ( $column == 'woopt' ) {
						$actions = get_post_meta( $postid, 'woopt_actions', true );

						if ( is_array( $actions ) && ( count( $actions ) > 0 ) ) {
							$running = false;

							foreach ( $actions as $action ) {
								$action_data = $this->woopt_action_data( $action );

								if ( ! empty( $action_data['conditional'] ) && $this->woopt_check_conditional( $action_data['conditional'] ) ) {
									$running = true;
								}
							}

							if ( $running ) {
								echo '<span class="woopt-icon running"><span class="dashicons dashicons-clock"></span></span>';
							} else {
								echo '<span class="woopt-icon"><span class="dashicons dashicons-clock"></span></span>';
							}
						}
					}
				}

				function woopt_add_export_column( $columns ) {
					$columns['woopt_actions'] = esc_html__( 'Timer', 'woopt' );

					return $columns;
				}

				function woopt_add_export_data( $value, $product ) {
					$value = get_post_meta( $product->get_id(), 'woopt_actions', true );

					return base64_encode( serialize( $value ) );
				}

				function woopt_add_column_to_importer( $options ) {
					$options['woopt_actions']       = esc_html__( 'Timer', 'woopt' );
					$options['woopt_unpurchasable'] = esc_html__( 'Unpurchasable', 'woopt' );

					return $options;
				}

				function woopt_add_column_to_mapping_screen( $columns ) {
					$columns['Timer']               = 'woopt_actions';
					$columns['timer']               = 'woopt_actions';
					$columns['woopt actions']       = 'woopt_actions';
					$columns['Unpurchasable']       = 'woopt_unpurchasable';
					$columns['unpurchasable']       = 'woopt_unpurchasable';
					$columns['woopt unpurchasable'] = 'woopt_unpurchasable';

					return $columns;
				}

				function woopt_process_import( $object, $data ) {
					if ( ! empty( $data['woopt_unpurchasable'] ) ) {
						// date time format 03/05/2020 05:35 am
						$timer = 'set_unpurchasable||date_time_after>' . $data['woopt_unpurchasable'];
						$object->update_meta_data( 'woopt_actions', array( $timer ) );
					} elseif ( ! empty( $data['woopt_actions'] ) ) {
						$object->update_meta_data( 'woopt_actions', unserialize( base64_decode( $data['woopt_actions'] ) ) );
					}

					return $object;
				}
			}

			new WPCleverWoopt();
		}
	}
} else {
	add_action( 'admin_notices', 'woopt_notice_premium' );
}

if ( ! function_exists( 'woopt_notice_wc' ) ) {
	function woopt_notice_wc() {
		?>
        <div class="error">
            <p><strong>WPC Product Timer</strong> require WooCommerce version 3.0.0 or greater.</p>
        </div>
		<?php
	}
}

if ( ! function_exists( 'woopt_notice_premium' ) ) {
	function woopt_notice_premium() {
		?>
        <div class="error">
            <p>Seems you're using both free and premium version of <strong>WPC Product Timer</strong>. Please
                deactivate the free version when using the premium version.</p>
        </div>
		<?php
	}
}