<?php
namespace Barn2\Plugin\WC_Restaurant_Ordering\Menu;

use Barn2\Plugin\WC_Restaurant_Ordering\Menu\Menu_Options,
    Barn2\Plugin\WC_Restaurant_Ordering\Template_Loader_Factory,
    Barn2\Plugin\WC_Restaurant_Ordering\Frontend_Scripts_Factory,
    Barn2\WRO_Lib\Template_Loader,
    WC_Product;

/**
 * Handles the display of a menu section. This is one grouping of products with an optional title and description.
 *
 * @package   Barn2\woocommerce-restaurant-ordering
 * @author    Barn2 Plugins <support@barn2.co.uk>
 * @license   GPL-3.0
 * @copyright Barn2 Media Ltd
 */
class Menu_Section {

    /**
     * @var string $title The section title.
     */
    protected $title;

    /**
     * @var string $description The section description.
     */
    protected $description;

    /**
     * @var WC_Product[] $products The products to display in this section.
     */
    protected $products;

    /**
     * @var Menu_Options $options The options for this section.
     */
    protected $options;

    /**
     * @var Template_Loader $template_loader The template loader.
     */
    private $template_loader;

    /**
     * @var Frontend_Scripts $scripts The frontend scripts object.
     */
    private $scripts;

    public function __construct( $title, $description, array $products, Menu_Options $options ) {
        $this->title       = (string) $title;
        $this->description = (string) $description;
        $this->products    = $this->parse_products( $products );
        $this->options     = $options;

        $this->template_loader = Template_Loader_Factory::create();
        $this->scripts         = Frontend_Scripts_Factory::create();
    }

    public function get_section() {
        $title_html = $this->template_loader->get_template( 'menu/section/title.php', [
            'title'       => $this->title,
            'description' => $this->description
            ] );

        $product_html = '';

        if ( ! empty( $this->products ) ) {
            foreach ( $this->products as $product ) {
                $menu_product = new Menu_Product( $product, $this->options );
                $product_html .= $menu_product->get_product();
            }
        }

        $products_html = $this->template_loader->get_template( 'menu/section/products.php', [
            'products'       => $product_html,
            'products_class' => $this->get_products_class()
            ] );

        $output = $this->template_loader->get_template( 'menu/section/section.php', [
            'title'    => $title_html,
            'products' => $products_html
            ] );

        $this->scripts->load_scripts();

        return $output;
    }

    protected function get_products_class() {
        $classes   = [ 'columns-' . $this->options->get_num_columns() ];
        $classes[] = 'image-' . $this->options->get_image_position();
        $classes[] = $this->options->show_product_image() ? 'show-image' : 'hide-image';
        $classes[] = $this->options->show_buy_button() ? 'not-clickable' : 'clickable';

        return implode( ' ', $classes );
    }

    private function parse_products( array $products ) {
        if ( empty( $products ) ) {
            return [];
        }

        $parsed = [];

        // Products can be passed as a list of WC_Product objects or product IDs.
        foreach ( $products as $product ) {
            if ( is_numeric( $product ) ) {
                $parsed[] = wc_get_product( $product );
            } elseif ( $product instanceof WC_Product ) {
                $parsed[] = $product;
            }
        }

        return array_filter( $parsed );
    }

}
