<?php
namespace Barn2\Plugin\WC_Restaurant_Ordering\Menu;

use Barn2\Plugin\WC_Restaurant_Ordering\Util;

/**
 * Stores the options for a restaurant order menu.
 *
 * @package   Barn2\woocommerce-restaurant-ordering
 * @author    Barn2 Plugins <support@barn2.co.uk>
 * @license   GPL-3.0
 * @copyright Barn2 Media Ltd
 */
class Menu_Options {

    const OT_QUICK    = 'quick';
    const OT_LIGHTBOX = 'lightbox';
    const OT_CHECK    = 'check';
    const DD_LIMITED  = 'limited';
    const DD_FULL     = 'full';
    const DEFAULTS    = [
        'categories'            => 'default',
        'order_type'            => self::OT_QUICK,
        'image_position'        => 'left',
        'product_image'         => true,
        'product_description'   => true,
        'description_length'    => self::DD_LIMITED,
        'buy_button'            => true,
        'category_titles'       => true,
        'category_descriptions' => true,
        'columns'               => 1
    ];

    /**
     * @var array $defaults
     */
    private static $defaults;

    /**
     * @var array $args
     */
    protected $args;

    /**
     * @var array $product_elements
     */
    protected $product_elements;

    public function __construct( array $args = [] ) {
        $this->args = $this->parse_args( $args );
    }

    public static function get_defaults() {
        if ( empty( self::$defaults ) ) {
            self::$defaults = array_merge( apply_filters( 'wc_restaurant_ordering_default_menu_options', self::DEFAULTS ), [
                'order_type'            => get_option( 'wro_order_type', self::DEFAULTS['order_type'] ),
                'image_position'        => get_option( 'wro_image_position', self::DEFAULTS['image_position'] ),
                'product_image'         => Util::get_checkbox_option( 'wro_show_product_image', self::DEFAULTS['product_image'] ),
                'product_description'   => Util::get_checkbox_option( 'wro_show_product_description', self::DEFAULTS['product_description'] ),
                'description_length'    => get_option( 'wro_description_length', self::DEFAULTS['description_length'] ),
                'buy_button'            => Util::get_checkbox_option( 'wro_show_buy_button', self::DEFAULTS['buy_button'] ),
                'category_titles'       => Util::get_checkbox_option( 'wro_show_category_titles', self::DEFAULTS['category_titles'] ),
                'category_descriptions' => Util::get_checkbox_option( 'wro_show_category_descriptions', self::DEFAULTS['category_descriptions'] ),
                'columns'               => get_option( 'wro_menu_columns', self::DEFAULTS['columns'] )
                ] );
        }
        return self::$defaults;
    }

    public function get_categories() {
        return $this->args['categories'] ? (array) $this->args['categories'] : [];
    }

    public function get_description_length() {
        return $this->args['description_length'];
    }

    public function get_image_position() {
        return $this->args['image_position'];
    }

    public function get_num_columns() {
        return absint( $this->args['columns'] );
    }

    public function get_order_type() {
        return $this->args['order_type'];
    }

    public function is_enabled( $data ) {
        return in_array( $data, $this->get_product_elements(), true );
    }

    public function show_category_titles() {
        return (bool) $this->args['category_titles'];
    }

    public function show_category_descriptions() {
        return (bool) $this->args['category_descriptions'];
    }

    public function show_product_image() {
        return (bool) $this->args['product_image'];
    }

    public function show_product_description() {
        return (bool) $this->args['product_description'];
    }

    public function show_buy_button() {
        return (bool) $this->args['buy_button'];
    }

    public function get_product_elements() {
        if ( empty( $this->product_elements ) ) {
            $elements = apply_filters( 'wc_restaurant_ordering_default_product_elements', [ 'name', 'price' ] );

            if ( $this->show_product_image() ) {
                $elements[] = 'image';
            }

            if ( $this->show_product_description() ) {
                $elements[] = 'description';
            }

            if ( $this->show_buy_button() ) {
                $elements[] = 'buy_button';
            }

            $this->product_elements = $elements;
        }

        return $this->product_elements;
    }

    private function parse_args( array $args ) {
        $defaults = self::get_defaults();

        // Remove any args not found in defaults.
        $args = array_intersect_key( array_merge( $defaults, $args ), $defaults );

        if ( empty( $args['categories'] ) || 'default' === $args['categories'] ) {
            $default_categories = (array) get_option( 'wro_categories', [] );

            if ( empty( $default_categories ) ) {
                // If no categories chosen in settings, fetch all categories.
                $default_categories = get_terms( [
                    'taxonomy' => 'product_cat',
                    'orderby'  => 'meta_value_num',
                    'meta_key' => 'order',
                    'fields'   => 'ids'
                    ] );

                $default_categories = ! is_wp_error( $default_categories ) ? (array) $default_categories : [];
            }

            $args['categories'] = $this->to_term_slugs( $default_categories );
        } elseif ( is_scalar( $args['categories'] ) ) {
            $args['categories'] = array_map( 'trim', explode( ',', strtolower( (string) $args['categories'] ) ) );
        }

        if ( ! in_array( $args['image_position'], [ 'left', 'right' ], true ) ) {
            $args['image_position'] = (string) $defaults['image_position'];
        }

        if ( ! in_array( $args['description_length'], [ self::DD_LIMITED, self::DD_FULL ], true ) ) {
            $args['description_length'] = (string) $defaults['description_length'];
        }

        if ( ! in_array( $args['order_type'], [ self::OT_QUICK, self::OT_LIGHTBOX ], true ) ) {
            $args['order_type'] = (string) $defaults['order_type'];
        }

        $args['columns'] = (int) $args['columns'];

        if ( ! in_array( $args['columns'], [ 1, 2, 3 ], true ) ) {
            $args['columns'] = (int) $defaults['columns'];
        }

        foreach ( [ 'category_titles', 'category_descriptions', 'product_image', 'product_description', 'buy_button' ] as $arg ) {
            $args[$arg] = ( 'true' === $args[$arg] || true === $args[$arg] ) ? true : false; ;
        }

        return $args;
    }

    private function to_term_slugs( array $category_ids ) {
        return array_filter(
            array_map( function( $term_id ) {
                $slug = get_term_field( 'slug', $term_id, 'product_cat' );
                return ! is_wp_error( $slug ) ? $slug : '';
            }, $category_ids )
        );
    }

}
