<?php
namespace Barn2\Plugin\WC_Restaurant_Ordering\Admin;

use Barn2\Plugin\WC_Restaurant_Ordering\Menu\Menu_Options,
    Barn2\Plugin\WC_Restaurant_Ordering\Util,
    Barn2\Plugin\WC_Restaurant_Ordering\Shortcodes,
    Barn2\WRO_Lib\Registerable,
    Barn2\WRO_Lib\Util as Lib_Util,
    Barn2\WRO_Lib\Plugin\Licensed_Plugin,
    Barn2\WRO_Lib\WooCommerce\Admin\Custom_Settings_Fields,
    WC_Settings_Page,
    WP_Post;

// Bail if WooCommerce settings page class not found.
if ( ! \class_exists( 'WC_Settings_Page' ) ) {
    return;
}

/**
 * The WooCommerce settings page. Appears under the main WooCommerce -> Settings menu.
 *
 * @package   Barn2\woocommerce-restaurant-ordering
 * @author    Barn2 Plugins <support@barn2.co.uk>
 * @license   GPL-3.0
 * @copyright Barn2 Media Ltd
 */
class Settings_Page extends WC_Settings_Page implements Registerable {

    /**
     * @var string The settings page ID.
     */
    protected $id = 'restaurant-ordering';

    /**
     * @var Licensed_Plugin The plugin object.
     */
    private $plugin;

    public function __construct( Licensed_Plugin $plugin ) {
        parent::__construct();

        $this->label  = __( 'Restaurant', 'woocommerce-restaurant-ordering' );
        $this->plugin = $plugin;
    }

    public function register() {
        $settings_fields = new Custom_Settings_Fields();
        $settings_fields->register();

        add_action( 'admin_enqueue_scripts', [ $this, 'load_scripts' ] );
        //add_action( 'woocommerce_settings_save_' . $this->id, [ $this, 'save_settings_saved' ], 20 );
    }

    public function load_scripts() {
        wp_enqueue_style( 'wro-settings', $this->plugin->get_dir_url() . 'assets/css/admin/settings.css', [], $this->plugin->get_version() );
        wp_enqueue_script( 'wro-settings', $this->plugin->get_dir_url() . 'assets/js/admin/settings.js', [ 'jquery', 'jquery-ui-sortable',
            'selectWoo', 'underscore' ], $this->plugin->get_version(), true );
    }

    public function get_settings() {
        $settings = [
            [
                'id'    => 'settings_start',
                'type'  => 'settings_start',
                'class' => 'restaurant-ordering-settings barn2-settings'
            ],
            [
                'name' => __( 'Restaurant Ordering', 'woocommerce-restaurant-ordering' ),
                'type' => 'title',
                'id'   => 'restaurant_ordering_section',
                /* translators: %s: the plugin name */
                'desc' => '<p>' . sprintf( __( 'The following options control the %s extension.', 'woocommerce-restaurant-ordering' ), $this->plugin->get_name() ) . '</p>' .
                '<p>' .
                Lib_Util::format_link( $this->plugin->get_documentation_url(), __( 'Documentation', 'woocommerce-restaurant-ordering' ) ) . ' | ' .
                Lib_Util::format_link( $this->plugin->get_support_url(), __( 'Support', 'woocommerce-restaurant-ordering' ) ) .
                '</p>'
            ],
            $this->plugin->get_license_setting()->get_license_key_setting(),
            $this->plugin->get_license_setting()->get_license_override_setting(),
            [
                'id'       => 'wro_menu_page',
                'type'     => 'single_select_page',
                'title'    => __( 'Restaurant order page', 'woocommerce-restaurant-ordering' ),
                'desc'     => __( 'The page to display your restaurant order form.', 'woocommerce-restaurant-ordering' ),
                'desc_tip' => __( 'You can also use the [restaurant_ordering] shortcode to show order forms on other pages.', 'woocommerce-restaurant-ordering' ),
                'autoload' => false
            ],
            [
                'id'                => 'wro_categories',
                'type'              => 'multiselect',
                'title'             => __( 'Categories', 'woocommerce-restaurant-ordering' ),
                'options'           => $this->get_product_categories_list(),
                'custom_attributes' => [
                    'aria-label'       => __( 'Categories', 'woocommerce-restaurant-ordering' ),
                    'data-placeholder' => __( 'Choose your menu categories&hellip;', 'woocommerce-restaurant-ordering' )
                ],
                'desc'              => __( 'The categories to include on your restaurant order page.', 'woocommerce-restaurant-ordering' ),
                'autoload'          => false
            ],
            [
                'id'            => 'wro_show_category_titles',
                'type'          => 'checkbox',
                'checkboxgroup' => 'start',
                'title'         => __( 'Order form options', 'woocommerce-restaurant-ordering' ),
                'desc'          => __( 'Show category titles.', 'woocommerce-restaurant-ordering' ),
                'default'       => Util::bool_to_checkbox_setting( Menu_Options::DEFAULTS['category_titles'] ),
                'autoload'      => false
            ],
            [
                'id'            => 'wro_show_category_descriptions',
                'type'          => 'checkbox',
                'checkboxgroup' => '',
                'desc'          => __( 'Show category descriptions.', 'woocommerce-restaurant-ordering' ),
                'default'       => Util::bool_to_checkbox_setting( Menu_Options::DEFAULTS['category_descriptions'] ),
                'autoload'      => false
            ],
            [
                'id'            => 'wro_show_product_image',
                'type'          => 'checkbox',
                'checkboxgroup' => '',
                'desc'          => __( 'Show product images.', 'woocommerce-restaurant-ordering' ),
                'default'       => Util::bool_to_checkbox_setting( Menu_Options::DEFAULTS['product_image'] ),
            ],
            [
                'id'            => 'wro_show_product_description',
                'type'          => 'checkbox_tooltip',
                'checkboxgroup' => '',
                'desc'          => __( 'Show product descriptions.', 'woocommerce-restaurant-ordering' ),
                'desc_tip'      => __( 'The short description is used if present, otherwise the long description is used.', 'woocommerce-restaurant-ordering' ),
                'default'       => Util::bool_to_checkbox_setting( Menu_Options::DEFAULTS['product_description'] ),
                'autoload'      => false
            ],
            [
                'id'            => 'wro_show_buy_button',
                'type'          => 'checkbox_tooltip',
                'checkboxgroup' => 'end',
                'desc'          => __( 'Show buy button.', 'woocommerce-restaurant-ordering' ),
                'desc_tip'      => __( 'If disabled, the product can be ordered by clicking to open a lightbox.', 'woocommerce-restaurant-ordering' ),
                'default'       => Util::bool_to_checkbox_setting( Menu_Options::DEFAULTS['buy_button'] ),
                'autoload'      => false
            ],
            [
                'id'      => 'wro_menu_columns',
                'type'    => 'select',
                'title'   => __( 'Number of columns', 'woocommerce-restaurant-ordering' ),
                'options' => [
                    1 => 1,
                    2 => 2,
                    3 => 3
                ],
                'css'     => 'max-width:85px',
                'default' => Menu_Options::DEFAULTS['columns']
            ],
            [
                'id'       => 'wro_image_position',
                'type'     => 'radio',
                'title'    => __( 'Image position', 'woocommerce-restaurant-ordering' ),
                'options'  => [
                    'left'  => __( 'Left', 'woocommerce-restaurant-ordering' ),
                    'right' => __( 'Right', 'woocommerce-restaurant-ordering' )
                ],
                'default'  => Menu_Options::DEFAULTS['image_position'],
                'autoload' => false
            ],
            [
                'id'       => 'wro_description_length',
                'type'     => 'radio',
                'title'    => __( 'Product description', 'woocommerce-restaurant-ordering' ),
                'options'  => [
                    Menu_Options::DD_LIMITED => __( 'Limit the description length', 'woocommerce-restaurant-ordering' ),
                    Menu_Options::DD_FULL    => __( 'Show full description', 'woocommerce-restaurant-ordering' )
                ],
                'desc_tip' => __( "By default, the description length in the order form is limited to fit the available space. If you select 'Show full description', the product box will expand as required.", 'woocommerce-restaurant-ordering' ),
                'default'  => Menu_Options::DEFAULTS['description_length'],
                'autoload' => false
            ],
            [
                'id'       => 'wro_order_type',
                'type'     => 'radio',
                'title'    => __( 'Order method', 'woocommerce-restaurant-ordering' ),
                'options'  => [
                    Menu_Options::OT_QUICK    => __( 'Quick', 'woocommerce-restaurant-ordering' ),
                    Menu_Options::OT_LIGHTBOX => __( 'Lightbox', 'woocommerce-restaurant-ordering' )
                ],
                'desc_tip' => __( 'Quick: add the product to the cart instantly if possible, otherwise open the lightbox to select options.', 'woocommerce-restaurant-ordering' ) . '<br/>' .
                __( 'Lightbox: always open the lightbox to order a product.', 'woocommerce-restaurant-ordering' ),
                'default'  => Menu_Options::DEFAULTS['order_type'],
                'autoload' => false
            ],
            [
                'id'            => 'wro_show_modal_image',
                'type'          => 'checkbox',
                'checkboxgroup' => 'start',
                'title'         => __( 'Lightbox options', 'woocommerce-restaurant-ordering' ),
                'desc'          => __( 'Show product image.', 'woocommerce-restaurant-ordering' ),
                'default'       => 'yes',
                'autoload'      => false
            ],
            [
                'id'            => 'wro_show_modal_description',
                'type'          => 'checkbox_tooltip',
                'checkboxgroup' => 'end',
                'desc'          => __( 'Show product description.', 'woocommerce-restaurant-ordering' ),
                'default'       => 'yes',
                'autoload'      => false,
                'desc_tip'      => __( 'Shows the long product description in the lightbox.', 'woocommerce-restaurant-ordering' ),
            ],
            [
                'type' => 'sectionend',
                'id'   => 'restaurant_ordering_section'
            ],
            [
                'id'   => 'settings_end',
                'type' => 'settings_end'
            ]
        ];

        return apply_filters( 'woocommerce_get_settings_' . $this->id, $settings );
    }

    public function save() {
        parent::save();

        if ( $order_page = get_option( 'wro_menu_page' ) ) {
            $order_page_post = get_post( $order_page );

            if ( $order_page_post instanceof WP_Post ) {
                $content = trim( $order_page_post->post_content );

                if ( false === strpos( $content, '[' . Shortcodes::MENU_SHORTCODE ) ) {
                    $shortcode_to_insert = '[' . Shortcodes::MENU_SHORTCODE . ']';

                    if ( $content ) {
                        $shortcode_to_insert = "\n\n" . $shortcode_to_insert;
                    }

                    $content .= $shortcode_to_insert;

                    wp_update_post( [
                        'ID'           => $order_page_post->ID,
                        'post_content' => $content
                    ] );
                }
            }
        }
    }

    private function get_product_categories_list() {
        $all_categories = get_terms( [
            'taxonomy'   => 'product_cat',
            'hide_empty' => false,
            'fields'     => 'id=>name'
            ] );

        $selected_term_ids   = (array) get_option( 'wro_categories', array() );
        $selected_categories = [];

        if ( $selected_term_ids ) {
            foreach ( $selected_term_ids as $term_id ) {
                // Check term still exists in the current categories list (it may have been deleted since options were last saved).
                if ( array_key_exists( $term_id, $all_categories ) ) {
                    $selected_categories[$term_id] = $all_categories[$term_id];
                }
            }
        }

        // Will append non-selected categories to the end of the list.
        $option_categories = $selected_categories + array_diff_key( $all_categories, $selected_categories );
        return $option_categories;
    }

}
