( function( $, window, document, params, undefined ) {
    'use strict';

    let formatError = function( status, error ) {
        status = status || '';
        error = error || '';

        if ( status && error ) {
            status = status + ': ';
        }
        if ( status || error ) {
            return '[' + status + error + ']';
        }
        return '';
    };

    let updateRestNonce = function( xhr ) {
        // Check for refreshed nonce returned from REST API and update our global param.
        const nonce = xhr.getResponseHeader( 'X-WP-Nonce' );

        if ( nonce ) {
            params.rest_nonce = nonce;
        }
    };

    let ProductModal = ( function() {

        const TEMPLATE_ID = 'wc-restaurant-product-modal';

        let currentModal = null,
            $htmlBody = $( 'html, body' );

        function Modal( productId ) {
            this.templateId = TEMPLATE_ID;

            if ( productId && 'object' === typeof productId ) {
                this.productData = productId;
                this.productId = parseInt( this.productData.product_id, 10 );
            } else {
                this.productId = parseInt( productId, 10 );
            }

            this.$modal = [];
            this.$cart = [];
            this.$quantity = [];
            this.$price = [];
            this.$addonsTotal = [];

            if ( !this.productId ) {
                if ( window.console ) {
                    window.console.warn( 'Invalid product ID provided for product modal.' );
                }
                return false;
            }
        }

        Modal.prototype.load = function() {
            if ( !$.isEmptyObject( this.productData ) ) {
                this.open();
                return;
            }

            let xhr = $.ajax( {
                url: params.rest_url + params.rest_endpoints.modal + '/' + this.productId,
                method: 'GET',
                dataType: 'json',
                cache: true,
                context: this,
                headers: {
                    'X-WP-Nonce': params.rest_nonce
                }
            } )
                .done( function( response, textStatus, xhr ) {
                    // Check for updated REST nonce.
                    updateRestNonce( xhr );

                    if ( response.success ) {
                        $( document.body ).trigger( 'wro:modal:load', response );

                        this.productData = response.product_data;
                        this.open();
                    } else {
                        RestaurantMenu.showCartNotice( response );
                    }
                } )
                .fail( function( xhr, textStatus, errorThrown ) {
                    if ( window.console ) {
                        window.console.error( 'An error occurred loading modal for product ID %i %s', this.productId, formatError( textStatus, errorThrown ) );
                    }
                    RestaurantMenu.showCartNotice( {
                        error_message: 'Sorry, this item cannot be loaded.'
                    } );
                } );

            return xhr;
        };

        Modal.prototype.open = function() {
            if ( $.isEmptyObject( this.productData ) ) {
                return;
            }

            // Close current modal first if there is one.
            close();

            $( '<div/>' ).WCBackboneModal( {
                template: this.templateId,
                variable: this.productData
            } );

            this.init();
            this.$modal.trigger( 'wro:modal:open' );
        };

        Modal.prototype.init = function() {
            this.$modal = getModalElement();

            if ( !this.$modal.length ) {
                return;
            }

            this.$cart = this.$modal.find( '.cart' );

            if ( this.$cart.length ) {
                this.$quantity = this.$cart.find( 'input.qty' );
                this.$price = this.$cart.find( '#product-total-' + this.productId );
                this.$addonsTotal = this.$cart.find( '#product-addons-total' );
            }

            this.bindEvents();
            this.initPlugins();
            this.initVariations();

            this.$modal.trigger( 'wro:modal:init' );
        };

        Modal.prototype.bindEvents = function() {
            this.$modal
                .off( '.restaurantOrder' )
                .on( 'wro:modal:open', onOpen )
                .on( 'wro:modal:close', onClose )
                .on( 'click.restaurantOrder', '.quantity .add', { modal: this }, onAddQuantity )
                .on( 'click.restaurantOrder', '.quantity .remove', { modal: this }, onRemoveQuantity )
                .on( 'change.restaurantOrder', '.quantity .qty', { modal: this }, onChangeQuantity )
                .on( 'wro:modal:update_price wro:modal:change_quantity', { modal: this }, onUpdatePrice )
                .on( 'updated_addons.restaurantOrder', '.cart', { modal: this }, onUpdateAddons )
                .on( 'wro:modal:before_add_product', { modal: this }, onBlock )
                .on( 'wro:modal:add_product_complete', { modal: this }, onUnblock )
                .on( 'submit.restaurantOrder', '.cart', { modal: this }, onAddToOrder )
                .on( 'focus.restaurantOrder', 'input, select, textarea', { modal: this }, onFocus );

            $( document )
                .on( 'wc_backbone_modal_before_remove.restaurantOrder', onBackboneClose );

            if ( this.hasProductAddons() ) {
                $( document )
                    .ajaxSend( onAjaxSend )
                    .ajaxComplete( onAjaxComplete );
            }
        };

        Modal.prototype.initPlugins = function() {
            let $modal = this.$modal;

            // WooCommerce Product Table.
            if ( $.fn.productTable ) {
                $modal.find( '.wc-product-table' ).productTable();
            }

            // WooCommerce Product Addons - we can initialise addons by triggering the 'quick-view-displayed' event.
            $modal.trigger( 'quick-view-displayed' );

            // WooCommerce Bundled Products.
            if ( $.fn.wc_pb_bundle_form ) {
                $modal.find( '.bundle_form .bundle_data' ).each( function() {

                    var $bundle_data = $( this ),
                        $composite_form = $bundle_data.closest( '.composite_form' );

                    // If part of a composite, let the composite initialize it.
                    if ( $composite_form.length === 0 ) {
                        $bundle_data.wc_pb_bundle_form();
                    }
                } );
            }

            // WooCommerce Composite Products.
            if ( $.fn.wc_composite_form ) {
                $modal.find( '.composite_form .composite_data' ).each( function() {
                    $( this ).wc_composite_form();
                } );
            }
        };

        Modal.prototype.initVariations = function() {
            if ( !$.fn.wc_variation_form ) {
                return;
            }

            let modal = this;

            // Initialise variations form.
            modal.$modal.find( 'form.variations_form' )
                .first()
                .wc_variation_form()
                .on( 'reset_data', function() {
                    modal.resetVariations();
                } )
                .on( 'found_variation', function( event, variation ) {
                    modal.setVariation( variation );
                } );
        };

        Modal.prototype.getQuantity = function() {
            if ( !this.$quantity.length ) {
                return 0;
            }

            let quantity = parseFloat( this.$quantity.val() );
            quantity = !isNaN( quantity ) ? quantity : 0;
            return quantity;
        };

        Modal.prototype.setQuantity = function( quantity ) {
            if ( !this.$quantity.length ) {
                return;
            }

            this.$quantity.val( quantity ).trigger( 'change' );
        };

        Modal.prototype.addRemoveQuantity = function( add ) {
            add = !!add;

            if ( !this.$quantity.length ) {
                return;
            }

            let quantity = parseFloat( this.$quantity.val() ),
                step = parseFloat( this.$quantity.prop( 'step' ) ),
                min = parseFloat( this.$quantity.prop( 'min' ) ),
                max = parseFloat( this.$quantity.prop( 'max' ) );

            // Sanitize quantity properties.
            step = !isNaN( step ) ? step : 1;
            min = !isNaN( min ) ? min : 1;
            max = !isNaN( max ) ? max : -1;

            // Ensure max is >= min.
            if ( max > 0 && max < min ) {
                max = min;
            }

            if ( isNaN( quantity ) ) {
                // If current quantity is invalid, just set to the minimum.
                quantity = min;
            } else if ( add ) {
                // Add quantity.
                quantity += step;

                if ( max > 0 ) {
                    quantity = window.Math.min( quantity, max );
                }
            } else {
                // Remove quantity.
                quantity -= step;
                quantity = window.Math.max( quantity, min );
            }

            this.setQuantity( quantity );
        };

        Modal.prototype.updatePrice = function() {
            // Bail if no price element found.
            if ( !this.$price.length ) {
                return;
            }

            let quantity = this.getQuantity(),
                itemPrice = parseFloat( this.$price.data( 'itemPrice' ) ),
                extrasPerItem = this.$price.data( 'extrasPerItem' ),
                extras = this.$price.data( 'extras' );

            if ( isNaN( itemPrice ) ) {
                return;
            }

            if ( !extrasPerItem ) {
                extrasPerItem = 0;
            }

            if ( !extras ) {
                extras = 0;
            }

            this.$price.html( formatPrice( extras + ( quantity * ( itemPrice + extrasPerItem ) ) ) );
        };

        Modal.prototype.setPrice = function( price ) {
            if ( this.$price.length ) {
                this.$price.data( 'itemPrice', price ).trigger( 'wro:modal:update_price' );
            }
        };

        Modal.prototype.setExtras = function( extras ) {
            if ( this.$price.length ) {
                this.$price.data( 'extras', extras ).trigger( 'wro:modal:update_price' );
            }
        };

        Modal.prototype.resetVariations = function() {
            // Disable add to cart button.
            this.$cart.find( '.order button, .order .qty' ).prop( 'disabled', true ).addClass( 'disabled' );

            let $variationIdInput = this.$cart.find( 'input[name="variation_id"]' );

            // Ensure 'variation_id' input is not blank - it must be 0 or a valid product ID, otherwise it breaks the product addons subtotal.
            if ( $variationIdInput.length && '' === $variationIdInput.val() ) {
                $variationIdInput.val( '0' );
            }
        };

        Modal.prototype.setVariation = function( variation ) {
            if ( !variation.is_purchasable || !variation.is_in_stock || !variation.variation_is_visible ) {

                if ( !variation.is_in_stock ) {
                    this.showOrderError( 'Sorry, this item is out of stock.' );
                } else {
                    this.showOrderError( 'Sorry, this item is not available right now.' );
                }
                return;
            }

            this.$cart.find( '.order button, .order .qty' ).prop( 'disabled', false ).removeClass( 'disabled' );

            if ( this.$quantity.length ) {
                if ( 'min_qty' in variation ) {
                    this.$quantity.prop( 'min', variation.min_qty );
                }

                if ( 'max_qty' in variation ) {
                    this.$quantity.prop( 'max', variation.max_qty );
                }
            }

            this.setPrice( variation.display_price );
        };

        Modal.prototype.removeOrderError = function() {
            this.$cart.find( '.order > .order-error' ).remove();
        };

        Modal.prototype.showOrderError = function( message ) {
            this.removeOrderError();
            $( '<p class="order-error">' + message + '</p>' ).prependTo( this.$cart.find( '.order' ) ).slideDown( 150 );
        };

        Modal.prototype.hideOrderError = function() {
            this.$cart.find( '.order > .order-error' ).slideUp( 150, function() {
                this.remove();
            } );
        };

        Modal.prototype.hasProductAddons = function() {
            return this.$addonsTotal.length && 'undefined' !== typeof woocommerce_addons_params;
        };

        Modal.prototype.block = function() {
            if ( this.$cart.length ) {
                this.$cart.find( '.buy' ).addClass( 'block' );
            }
        };

        Modal.prototype.unblock = function() {
            if ( this.$cart.length ) {
                this.$cart.find( '.buy' ).removeClass( 'block' );
            }
        };

        function close( $modal ) {
            $modal = $modal || getModalElement();

            if ( !$modal.length ) {
                return;
            }

            // We can't access the Backbone View to call view.remove(), so we simulate a ESC key press to close the modal.
            let keyPress = $.Event( 'keydown' );
            keyPress.which = 27; // ESC key
            $modal.trigger( keyPress );
        }

        function formatPrice( amount ) {
            if ( 'function' !== typeof window.accounting.formatMoney ) {
                return amount;
            }

            return window.accounting.formatMoney( amount, {
                symbol: params.price_currency_symbol,
                decimal: params.price_decimal_sep,
                thousand: params.price_thousand_sep,
                precision: params.price_num_decimals,
                format: params.price_currency_format
            } );
        }

        function getCurrentModal() {
            return currentModal;
        }

        function getFormData( $form ) {
            var data = { };

            $.each( $form.serializeArray(), function( index, item ) {
                if ( item.name.indexOf( '[]' ) !== -1 ) {
                    item.name = item.name.replace( '[]', '' );
                    data[ item.name ] = $.makeArray( data[ item.name ] );
                    data[ item.name ].push( item.value );
                } else {
                    data[ item.name ] = item.value;
                }
            } );

            return data;
        }

        function getModalElement() {
            return $( document.body ).children( '#wc-backbone-modal-dialog' );
        }

        function load( productId ) {
            currentModal = new Modal( productId );
            return currentModal.load();
        }

        function show( productData ) {
            currentModal = new Modal( productData );
            currentModal.open();
        }

        function setPrice( price ) {
            if ( currentModal ) {
                currentModal.setPrice( price );
            }
        }

        function setExtras( extras ) {
            if ( currentModal ) {
                currentModal.setExtras( extras );
            }
        }

        function validateOrder( $cartForm ) {
            let formData = getFormData( $cartForm );

            let result = {
                'isValid': false,
                'message': false
            };

            if ( $.isEmptyObject( formData ) || !( 'product_id' in formData ) ) {
                result.message = 'Sorry, there was an error adding this product.';
                return result;
            }

            // Check quantity is valid.
            if ( !( 'quantity' in formData ) ) {
                formData.quantity = 1;
            }

            if ( !formData.quantity ) {
                result.message = 'Please enter a quantity greater than 0.';
                return result;
            }

            // Check variation has been selected.
            if ( 'variation_id' in formData && ( 0 === parseInt( formData.variation_id, 10 ) ) ) {
                result.message = 'Please select all required options.';
                return result;
            }

            // Validate required options/addons.
            $( 'select, input, textarea', $cartForm ).each( function() {
                let name = $( this ).prop( 'name' );

                // Strip [] from input name if present, to match properties in formData.
                if ( -1 !== name.indexOf( '[]' ) ) {
                    name = name.replace( '[]', '' );
                }

                if ( $( this ).prop( 'required' ) ) {
                    if ( !( name in formData ) || '' === formData[name] ) {
                        result.message = 'Please select all required options.';
                        return result;
                    }
                }
            } );

            if ( !result.message ) {
                result.isValid = true;
            }

            result.formData = formData;
            return result;
        }

        // EVENTS

        function onAddQuantity( event ) {
            event.data.modal.addRemoveQuantity( true );
            event.data.modal.$modal.trigger( 'wro:modal:add_quantity' );
        }

        function onRemoveQuantity( event ) {
            event.data.modal.addRemoveQuantity( false );
            event.data.modal.$modal.trigger( 'wro:modal:remove_quantity' );
        }

        function onFocus( event ) {
            event.data.modal.hideOrderError();
        }

        function onChangeQuantity( event ) {
            // Skip if product has Product Addons as 'onAddonsUpdated' will handle it.
            if ( event.data.modal.hasProductAddons() ) {
                return true;
            }

            event.data.modal.$modal.trigger( 'wro:modal:change_quantity' );
        }

        function onUpdateAddons( event ) {
            let modal = event.data.modal,
                addonsData = modal.$addonsTotal.data( 'price_data' ),
                $price = modal.$price,
                addonsTotal = 0;

            if ( !Array.isArray( addonsData ) || !$price.length ) {
                return true;
            }

            if ( addonsData.length ) {
                addonsTotal = addonsData.reduce( function( runningTotal, addon ) {
                    if ( !( 'cost_raw' in addon ) ) {
                        return runningTotal;
                    }

                    let addonCost = parseFloat( addon.cost_raw );
                    return !isNaN( addonCost ) ? ( runningTotal + addonCost ) : runningTotal;
                }, addonsTotal );
            }

            modal.setExtras( addonsTotal );
        }

        function onUpdatePrice( event ) {
            event.data.modal.updatePrice();
        }

        function onAddToOrder( event ) {
            let $cartForm = $( event.target ),
                modal = event.data.modal,
                $modal = modal.$modal;

            modal.removeOrderError();
            let formValidation = validateOrder( $cartForm );

            if ( !formValidation.isValid ) {
                modal.showOrderError( formValidation.message );
                return false;
            }

            $modal.trigger( 'wro:modal:before_add_product', [formValidation.formData] );

            $.ajax( {
                url: params.rest_url + params.rest_endpoints.cart,
                type: 'POST',
                data: formValidation.formData,
                dataType: 'json',
                headers: {
                    'X-WP-Nonce': params.rest_nonce
                }
            } )
                .done( function( response, status, xhr ) {
                    // Triggering wro:add_product event will display success message and refresh cart widget.
                    $modal.trigger( 'wro:add_product', [response] );
                    $modal.trigger( 'wro:modal:add_product', [response] );
                } )
                .fail( function( xhr, textStatus, errorThrown ) {
                    if ( window.console ) {
                        window.console.error( 'Error adding product to cart ' + formatError( textStatus, errorThrown ) );
                    }
                    $modal.trigger( 'wro:add_product_fail' ).trigger( 'wro:modal:add_product_fail' );
                } )
                .always( function() {
                    $modal.trigger( 'wro:modal:add_product_complete' );
                    close( $modal );
                } );

            return false;
        }

        function onBackboneClose( event, target ) {
            if ( TEMPLATE_ID !== target ) {
                return true;
            }

            let $modal = getModalElement();

            if ( !$modal.length ) {
                return true;
            }

            $modal.trigger( 'wro:modal:close' );
        }

        function onOpen() {
            // Some themes set overflow on <html> so we need to remove this while modal is open.
            $htmlBody.addClass( 'wc-restaurant-modal-active' );
        }

        function onClose() {
            // Reset <html> overflow on close.
            $htmlBody.removeClass( 'wc-restaurant-modal-active' );
        }

        function onBlock( event ) {
            event.data.modal.block();
        }

        function onUnblock( event ) {
            event.data.modal.unblock();
        }

        function onAjaxSend( event, jqxhr, settings ) {
            // We need to block the modal price if Product Addons does an Ajax request to recalculate totals and taxes.
            // The only way to do this is to check the data.action passed in Ajax object.
            if ( currentModal && 'data' in settings && settings.data.indexOf( 'action=wc_product_addons_calculate_tax' ) >= 0 ) {
                currentModal.block();
            }
        }

        function onAjaxComplete( event, jqxhr, settings ) {
            if ( currentModal ) {
                currentModal.unblock();
            }
        }

        // Return product modal API.
        return {
            load: load,
            show: show,
            close: close,
            getCurrentModal: getCurrentModal,
            getModalElement: getModalElement,
            setExtras: setExtras,
            setPrice: setPrice
        };

    } )();

    let RestaurantMenu = ( function() {
        let cartNoticeTemplate = null,
            $sections = [];

        function init() {
            $sections = $( '.wc-restaurant-menu-section' );

            bindEvents();
            initCartNotice();
        }

        function bindEvents() {
            $sections
                .off( '.restaurantOrder' )
                .on( 'click.restaurantOrder', '.clickable .wc-restaurant-menu-product', onClickProduct )
                .on( 'click.restaurantOrder', '.wc-restaurant-menu-product .add', onAddProduct )
                .on( 'click.restaurantOrder', '.wc-restaurant-menu-product .remove', onRemoveProduct );

            $( document.body )
                .on( 'wro:add_product', onProductAdded )
                .on( 'wro:add_product_fail', onAddProductFail );
        }

        function initCartNotice() {
            cartNoticeTemplate = wp.template( 'wc-restaurant-cart-notice' );
        }

        function showCartNotice( data ) {
            if ( !cartNoticeTemplate ) {
                return;
            }

            data = $.extend( { success: false, error_message: '' }, data );

            let $notice = $( cartNoticeTemplate( data ) );
            $notice.appendTo( document.body ).slideDown();

            setTimeout( function() {
                $notice.fadeOut( function() {
                    this.remove();
                } );
            }, 2700 );
        }

        function checkOrderType( productId ) {
            productId = parseInt( productId, 10 );

            $( document.body ).trigger( 'wro:before_check_order_type', productId );

            let xhr = $.ajax( {
                url: params.rest_url + params.rest_endpoints.order_type + '/' + productId,
                method: 'GET',
                dataType: 'json',
                cache: true,
                headers: {
                    'X-WP-Nonce': params.rest_nonce
                }
            } )
                .done( function( response, textStatus, xhr ) {
                    // Check for updated REST nonce.
                    updateRestNonce( xhr );

                    $( document.body ).trigger( 'wro:check_order_type', [response] );
                } )
                .always( function() {
                    $( document.body ).trigger( 'wro:check_order_type_complete' );
                } )
                .fail( function( xhr, textStatus, errorThrown ) {
                    if ( window.console ) {
                        window.console.error( 'An error occurred checking the order type for product ID %i %s', productId, formatError( textStatus, errorThrown ) );
                    }
                    $( document.body ).trigger( 'wro:check_order_type_fail' );
                } );

            return xhr;
        }

        function quickAddRemoveProduct( productId, quantity, action ) {
            productId = parseInt( productId, 10 );
            quantity = parseFloat( quantity );

            // We must have a product ID and a quantity greater than 0 to add or remove.
            if ( !productId || !quantity ) {
                return false;
            }

            action = ( 'remove' === action ) ? action : 'add';

            const method = ( 'remove' === action ) ? 'DELETE' : 'POST',
                cartData = {
                    'product_id': productId,
                    'quantity': quantity
                };

            $( document.body ).trigger( 'wro:before_add_product', [cartData] );

            let xhr = $.ajax( {
                url: params.rest_url + params.rest_endpoints.cart,
                method: method,
                data: cartData,
                dataType: 'json',
                headers: {
                    'X-WP-Nonce': params.rest_nonce
                }
            } )
                .done( function( response, status, xhr ) {
                    // Check for updated REST nonce.
                    updateRestNonce( xhr );

                    // Trigger add/remove events.
                    $( document.body )
                        .trigger( `wro:${action}_product`, response )
                        .trigger( `wro:quick_${action}_product`, response );
                } )
                .always( function() {
                    $( document.body ).trigger( 'wro:add_product_complete' );
                } )
                .fail( function( xhr, textStatus, errorThrown ) {
                    if ( window.console ) {
                        window.console.error( 'An error occurred - %s product ID %i %s.', action, productId, formatError( textStatus, errorThrown ) );
                    }
                    $( document.body ).trigger( 'wro:add_product_fail' );
                } );

            return xhr;
        }

        function block( $el ) {
            $el.addClass( 'block' );
        }

        function unblock( $el ) {
            $el.removeClass( 'block' );
        }

        function onClickProduct( event ) {
            let $product = $( event.currentTarget ),
                productId = $product.data( 'productId' ) || 0;

            // Bail if we couldn't find a product ID.
            if ( !productId ) {
                return true;
            }

            let $price = $( '.price', $product );
            block( $price );

            ProductModal.load( productId )
                .always( function() {
                    unblock( $price );
                } );

            return false;
        }

        function onAddProduct( event ) {
            let $target = $( event.target ),
                $product = $target.parents( '.wc-restaurant-menu-product' ),
                productId = $product.data( 'productId' ) || 0,
                orderType = $product.data( 'orderType' ) || false,
                quantity = 1;

            // Bail if we couldn't find a product ID.
            if ( !productId ) {
                return true;
            }

            block( $target );

            let xhr = false,
                unblockOnComplete = true;

            if ( 'check' === orderType ) {
                xhr = checkOrderType( productId )
                    .done( function( response, status, xhr ) {
                        if ( !response.success ) {
                            showCartNotice( response );
                            return;
                        }

                        // The returned order_type should be 'quick' or 'lightbox'. Update the product data with this.
                        $product.data( 'orderType', response.order_type );

                        // If product can be quick added, do that, otherwise open the modal.
                        if ( 'quick' === response.order_type ) {
                            // Prevent spinner removal below, instead do it after quickAddRemoveProduct.
                            unblockOnComplete = false;

                            quickAddRemoveProduct( productId, quantity, 'add' )
                                .always( function() {
                                    unblock( $target );
                                } );
                        } else if ( 'product_data' in response ) {
                            // Modal required, so load this with the returned product data.
                            ProductModal.show( response.product_data );
                        }
                    } );
            } else if ( 'quick' === orderType ) {
                xhr = quickAddRemoveProduct( productId, quantity, 'add' );
            } else {
                // Otherwise must be lightbox.
                // Prevent multiple clicks loading more than one modal.
                if ( $target.data( 'modal' ) ) {
                    return false;
                }
                $target.data( 'modal', true );

                // Load lightbox for this product.
                xhr = ProductModal.load( productId );
            }

            if ( xhr ) {
                xhr.always( function() {
                    $target.data( 'modal', false );

                    if ( unblockOnComplete ) {
                        unblock( $target );
                    }
                } );
            }

            return false;
        }

        function onRemoveProduct( event ) {
            const $product = $( event.target ).parents( '.wc-restaurant-menu-product' ),
                productId = $product.data( 'productId' ) || 0,
                quantity = 1;

            // Bail if we couldn't find a product ID.
            if ( !productId ) {
                return true;
            }

            quickAddRemoveProduct( productId, quantity, 'remove' );
            return false;
        }

        function onProductAdded( event, response ) {
            // Always show cart errors, regardless of show_cart_notice param.
            if ( !response.success ) {
                showCartNotice( response );
                return;
            }

            if ( params.show_cart_notice ) {
                showCartNotice( response );
            }

            if ( params.refresh_cart && 'fragments' in response && 'cart_hash' in response ) {
                // Trigger the WooCommerce 'added_to_cart' event so themes can refresh cart fragments.
                $( event.target ).trigger( 'added_to_cart', [response.fragments, response.cart_hash] );
            }
        }

        function onAddProductFail() {
            showCartNotice( { success: false } );
        }

        // Return restuarant menu API.
        return {
            init: init,
            addRemoveProduct: quickAddRemoveProduct,
            checkOrderType: checkOrderType,
            showCartNotice: showCartNotice
        };

    } )();

    // Expose APIs on global object.
    window.wcRestaurantMenu = RestaurantMenu;
    window.wcRestaurantProductModal = ProductModal;

    $( document ).ready( function( ) {
        RestaurantMenu.init();
    } );

}
)( jQuery, window, document, wc_restaurant_ordering_params );
