<?php
namespace Barn2\Plugin\WC_Restaurant_Ordering;

use Barn2\Plugin\WC_Restaurant_Ordering\Admin\Plugin_Setup,
    Barn2\Plugin\WC_Restaurant_Ordering\Admin\Admin_Controller,
    Barn2\Plugin\WC_Restaurant_Ordering\Rest\Rest_Controller,
    Barn2\Plugin\WC_Restaurant_Ordering\Shortcodes,
    Barn2\Plugin\WC_Restaurant_Ordering\Frontend_Scripts,
    Barn2\WRO_Lib\Registerable,
    Barn2\WRO_Lib\Translatable,
    Barn2\WRO_Lib\Service_Provider,
    Barn2\WRO_Lib\Plugin\Premium_Plugin,
    Barn2\WRO_Lib\Plugin\Licensed_Plugin,
    Barn2\WRO_Lib\Util,
    Barn2\WRO_Lib\Admin\Notices;

/**
 * The main plugin class. Responsible for setting up to core plugin services.
 *
 * @package   Barn2\woocommerce-restaurant-ordering
 * @author    Barn2 Plugins <support@barn2.co.uk>
 * @license   GPL-3.0
 * @copyright Barn2 Media Ltd
 */
class Plugin extends Premium_Plugin implements Registerable, Translatable, Service_Provider, Licensed_Plugin {

    const NAME    = 'WooCommerce Restaurant Ordering';
    const ITEM_ID = 236796;

    private $services = [];

    /**
     * Constructs and initalizes the main plugin class.
     *
     * @param string $file The path to the main plugin file.
     * @param string $version The current plugin version.
     */
    public function __construct( $file = null, $version = '1.0' ) {
        parent::__construct( [
            'name'               => self::NAME,
            'item_id'            => self::ITEM_ID,
            'version'            => $version,
            'file'               => $file,
            'is_woocommerce'     => true,
            'settings_path'      => 'admin.php?page=wc-settings&tab=restaurant-ordering',
            'documentation_path' => 'kb-categories/wro-kb'
        ] );
    }

    /**
     * Registers the plugin with WordPress.
     */
    public function register() {
        parent::register();

        add_action( 'init', [ $this, 'load_textdomain' ], 5 );
        add_action( 'init', [ $this, 'load_services' ] );

        // We create Plugin_Setup here so the plugin activation hook will run.
        $plugin_setup = new Plugin_Setup( $this->get_file() );
        $plugin_setup->register();
    }

    public function load_services() {
        // Don't load anything if WooCommerce not active.
        if ( ! Util::is_woocommerce_active() ) {
            $this->add_missing_woocommerce_notice();
            return;
        }

        $this->services = [
            'admin' => new Admin_Controller( $this )
        ];

        // Create core services if license is valid.
        if ( $this->has_valid_license() ) {
            $this->services['rest_controller']  = new Rest_Controller();
            $this->services['shortcodes']       = new Shortcodes();
            $this->services['frontend_scripts'] = new Frontend_Scripts( $this->get_dir_url(), $this->get_version(), $this->services['rest_controller'] );

            foreach ( $this->get_integrations() as $integration => $class ) {
                $this->services[$integration] = new $class();
            }
        }

        Util::register_services( $this->services );
    }

    public function get_service( $id ) {
        return isset( $this->services[$id] ) ? $this->services[$id] : null;
    }

    public function get_services() {
        return $this->services;
    }

    public function load_textdomain() {
        load_plugin_textdomain( 'woocommerce-restaurant-ordering', false, $this->get_slug() . '/languages' );
    }

    protected function get_integrations() {
        return [
            'product_table' => Integrations\Product_Table::class
        ];
    }

    private function add_missing_woocommerce_notice() {
        if ( Util::is_admin() ) {
            $admin_notice = new Notices();
            $admin_notice->add(
                'wro_woocommerce_missing',
                '',
                sprintf(
                    /* translators: %1$s: open link tag, %2$s: close link tag, %3$s: the plugin name */
                    __( 'Please %1$sinstall WooCommerce%2$s in order to use the %3$s extension.', 'woocommerce-restaurant-ordering' ),
                    Util::format_link_open( 'https://woocommerce.com/', true ),
                    '</a>',
                    $this->get_name() ),
                array(
                    'type'       => 'error',
                    'capability' => 'install_plugins'
            ) );
            $admin_notice->boot();
        }
    }

}
