<?php
namespace Barn2\Plugin\WC_Restaurant_Ordering\Menu;

use Barn2\Plugin\WC_Restaurant_Ordering\Menu\Menu_Options,
    Barn2\Plugin\WC_Restaurant_Ordering\Template_Loader_Factory,
    Barn2\WRO_Lib\Template_Loader,
    Barn2\WRO_Lib\Util;

/**
 * Handles the display of a restaurant menu for a given set of options.
 *
 * @package   Barn2\woocommerce-restaurant-ordering
 * @author    Barn2 Plugins <support@barn2.co.uk>
 * @license   GPL-3.0
 * @copyright Barn2 Media Ltd
 */
class Restaurant_Menu {

    /**
     * @var Menu_Options $options The menu options.
     */
    protected $options;

    /**
     * @var Template_Loader $template_loader The template loader.
     */
    private $template_loader;

    public function __construct( Menu_Options $options ) {
        $this->options         = $options;
        $this->template_loader = Template_Loader_Factory::create();
    }

    public function get_menu() {
        if ( ! Util::is_woocommerce_active() ) {
            return $this->error_message( __( 'The Restaurant Ordering extension requires WooCommerce to be installed', 'woocommerce-restaurant-ordering' ) );
        }

        $sections = [];

        foreach ( $this->options->get_categories() as $category_slug ) {
            $term = get_term_by( 'slug', $category_slug, 'product_cat' );

            if ( $term ) {
                $sections[$category_slug] = $term;
            }
        }

        if ( empty( $sections ) ) {
            return $this->error_message( __( 'There are no valid categories available for this menu', 'woocommerce-restaurant-ordering' ), true );
        }

        $products      = $this->load_products( array_keys( $sections ) );
        $sections_html = '';

        foreach ( $sections as $category_slug => $term ) {
            $section_title       = $this->options->show_category_titles() ? get_term_field( 'name', $term, 'product_cat' ) : '';
            $section_description = $this->options->show_category_descriptions() ? trim( get_term_field( 'description', $term, 'product_cat' ) ) : '';
            $section_products    = [];

            foreach ( $products as $product ) {
                if ( in_array( $term->term_id, $product->get_category_ids() ) ) {
                    $section_products[] = $product;
                }
            }

            // Run section data through filters.
            $section_title       = apply_filters( 'wc_restaurant_ordering_category_title', $section_title, $term );
            $section_description = apply_filters( 'wc_restaurant_ordering_category_description', $section_description, $term );
            $section_products    = apply_filters( 'wc_restaurant_ordering_category_products', $section_products, $term );

            // Add this section if we have 1 or more products.
            if ( ! empty( $section_products ) ) {
                $menu_section  = new Menu_Section( $section_title, $section_description, $section_products, $this->options );
                $sections_html .= $menu_section->get_section();
            }
        }

        if ( ! $sections_html ) {
            return $this->error_message( __( 'There are no products available for this menu', 'woocommerce-restaurant-ordering' ), true );
        }

        $output = $this->template_loader->get_template( 'menu/menu.php', [ 'sections' => $sections_html ] );
        return $output;
    }

    protected function error_message( $message, $show_categories = false ) {
        $messge = esc_html( $message );

        if ( $show_categories ) {
            $message .= ' ' . sprintf( __( '[categories: %s]', 'woocommerce-restaurant-ordering' ), implode( ', ', $this->options->get_categories() ) );
        }

        return '<p class="wc-restaurant-menu-error">' . $message . '</p>';
    }

    private function load_products( array $categories ) {
        $wc_get_products_args = [
            'category' => $categories,
            'limit'    => apply_filters( 'wc_restaurant_ordering_product_limit', 250 ),
            'status'   => 'publish',
            'type'     => apply_filters( 'wc_restaurant_ordering_allowed_product_types', [ 'simple', 'variable' ] )
        ];

        if ( ! apply_filters( 'wc_restaurant_ordering_show_hidden_products', false ) ) {
            $wc_get_products_args['visibility'] = 'visible';
        }

        if ( 'yes' === get_option( 'woocommerce_hide_out_of_stock_items' ) ) {
            $wc_get_products_args['stock_status'] = 'instock';
        }

        return wc_get_products( apply_filters( 'wc_restaurant_ordering_wc_get_products_args', $wc_get_products_args, $this->options ) );
    }

}
