<?php
namespace Barn2\Plugin\WC_Restaurant_Ordering\Menu;

use Barn2\Plugin\WC_Restaurant_Ordering\Menu\Product_Data,
    Barn2\Plugin\WC_Restaurant_Ordering\Util,
    Barn2\Plugin\WC_Restaurant_Ordering\Template_Loader_Factory,
    Barn2\WRO_Lib\Template_Loader,
    Barn2\WRO_Lib\Util as Lib_Util,
    WC_Product;

/**
 * Handles retrieval of the modal data for a product.
 *
 * @package   Barn2\woocommerce-restaurant-ordering
 * @author    Barn2 Plugins <support@barn2.co.uk>
 * @license   GPL-3.0
 * @copyright Barn2 Media Ltd
 */
class Product_Modal {

    /**
     * @var WC_Product The product for this modal.
     */
    private $product;

    /**
     * @var Template_Loader $template_loader The template loader.
     */
    private $template_loader;

    public function __construct( WC_Product $product ) {
        $this->product         = $product;
        $this->template_loader = Template_Loader_Factory::create();
    }

    /**
     * Get the modal data for a product.
     *
     * @return array The modal data.
     * @global WC_Product $product Global product.
     */
    public function get_modal_data() {
        global $product;

        // Store current global product, then update global with our modal product.
        $old_product = $product;
        $product     = $this->product;

        // Build the modal data.
        $data = [
            'product_id'    => esc_attr( $product->get_id() ),
            'product_name'  => apply_filters( 'wc_restuarant_ordering_product_name', $product->get_name(), $product ),
            'price'         => esc_attr( $product->get_price() ),
            'display_price' => apply_filters( 'wc_restaurant_ordering_modal_product_price', Product_Data::get_display_price( $product, false, false ), $product ),
            'quantity'      => $this->get_modal_quantity_args( $product ),
            'in_stock'      => $product->is_in_stock(),
            'form_class'    => 'cart',
            'form_data'     => '',
            'options'       => ''
        ];

        if ( Util::get_checkbox_option( 'wro_show_modal_image', true ) ) {
            $data['image'] = apply_filters( 'wc_restaurant_ordering_modal_product_image', Product_Data::get_image_url( $product, $this->get_modal_image_size() ), $product );
        }

        if ( Util::get_checkbox_option( 'wro_show_modal_description', true ) ) {
            $data['description'] = $this->get_modal_description();
        }

        // Handle variable products.
        if ( $product->is_type( 'variable' ) ) {
            $available_variations = $product->get_available_variations();

            if ( ! empty( $available_variations ) ) {
                $data['options'] = $this->template_loader->get_template( 'modal/variations.php', [
                    'product'              => $product,
                    'variation_attributes' => $product->get_variation_attributes()
                    ] );
                //todo: add variations reset link?

                $data['form_class'] .= ' variations_form';
                $data['form_data']  .= sprintf( ' data-product_variations="%s"', wc_esc_json( wp_json_encode( $available_variations ) ) );
            } else {
                // Out of stock (no available variations).
                $data['in_stock'] = false;
            }
        }

        // Handle product addons.
        if ( Lib_Util::is_product_addons_active() && ! empty( $GLOBALS['Product_Addon_Display'] ) ) {
            ob_start();
            $GLOBALS['Product_Addon_Display']->display( $product->get_id() );
            $data['options'] .= str_replace( [ "\n", "\t" ], '', ob_get_clean() );
        }

        // Run data through a filter.
        $data = apply_filters( 'wc_restaurant_ordering_modal_data', $data, $product );

        // Trim the string data.
        $data = array_map( function( $value ) {
            return is_string( $value ) ? trim( $value ) : $value;
        }, $data );

        // Reset the global product.
        $product = $old_product;

        return $data;
    }

    protected function get_modal_description() {
        $description = '';

        // Fetch long description for product by default, but allow themes/plugins to override.
        if ( apply_filters( 'wc_restaurant_ordering_modal_use_long_description', true, $this->product ) ) {
            $description = $this->product->get_description();
        }

        if ( ! $description && apply_filters( 'wc_restaurant_ordering_modal_use_short_description', true, $this->product ) ) {
            $description = $this->product->get_short_description();
        }

        return apply_filters( 'wc_restaurant_ordering_modal_description', apply_filters( 'the_content', $description ), $this->product );
    }

    protected function get_modal_image_size() {
        return apply_filters( 'wc_restaurant_ordering_modal_image_size', [ 500, 300 ], $this->product );
    }

    private function get_modal_quantity_args( WC_Product $product ) {
        // Create the quantity args using default WC names and run them through the WC filter to allow plugins to modify.
        $quantity_args = Product_Data::get_quantity_args( $product );

        // Min quantity must be greater >= 0.
        $quantity_args['min_value'] = max( $quantity_args['min_value'], 0 );

        if ( -1 === $quantity_args['max_value'] ) {
            // A max of -1 means unlimited, so set to high number.
            $quantity_args['max_value'] = 999999;
        } elseif ( $quantity_args['max_value'] < $quantity_args['min_value'] ) {
            // Max can't be less than min.
            $quantity_args['max_value'] = $quantity_args['min_value'];
        }

        $result = [];
        $map    = [
            'input_value' => 'value',
            'min_value'   => 'min',
            'max_value'   => 'max',
            'step'        => 'step'
        ];

        foreach ( $map as $wc_key => $key ) {
            $result[$key] = esc_attr( $quantity_args[$wc_key] );
        }

        return $result;
    }

}
