<?php
/**
 * WooCommerce First Data
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade WooCommerce First Data to newer
 * versions in the future. If you wish to customize WooCommerce First Data for your
 * needs please refer to http://docs.woocommerce.com/document/firstdata/
 *
 * @author      SkyVerge
 * @copyright   Copyright (c) 2013-2021, SkyVerge, Inc.
 * @license     http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

namespace SkyVerge\WooCommerce\First_Data\Payeezy\API\Response\PaymentJS;

defined( 'ABSPATH' ) or exit;

use SkyVerge\WooCommerce\PluginFramework\v5_10_3 as Framework;
use SkyVerge\WooCommerce\First_Data\Payeezy\API\Response;

/**
 * Payeezy Payment.JS API Authorize Session response.
 *
 * @since 4.7.0
 */
class Authorize_Session extends Response\PaymentJS {


	/** @var string the client token generated by the request */
	protected $client_token = '';

	/** @var string the nonce generated by the request */
	protected $nonce = '';


	/** Getter methods ************************************************************************************************/


	/**
	 * Gets the client token.
	 *
	 * @since 4.7.0
	 *
	 * @return string
	 */
	public function get_client_token() {

		return $this->client_token;
	}


	/**
	 * Gets the nonce.
	 *
	 * @since 4.7.0
	 *
	 * @return string
	 */
	public function get_nonce() {

		return $this->nonce;
	}


	/**
	 * Gets the public key.
	 *
	 * @since 4.7.0
	 *
	 * @return string
	 */
	public function get_public_key() {

		return isset( $this->response_data->publicKeyBase64 ) ? $this->response_data->publicKeyBase64 : '';
	}


	/**
	 * Gets an error message, if any.
	 *
	 * @since 4.7.0
	 *
	 * @return string
	 */
	public function get_error() {

		return isset( $this->response_data->error ) ? $this->response_data->error : '';
	}


	/** Setter methods ************************************************************************************************/


	/**
	 * Sets the response's client token.
	 *
	 * This is necessary because First Data doesn't send it along in the response data, but rather a header.
	 *
	 * @since 4.7.0
	 *
	 * @param string $value token value
	 * @throws Framework\SV_WC_API_Exception
	 */
	public function set_client_token( $value ) {

		if ( ! is_string( $value ) ) {
			throw new Framework\SV_WC_API_Exception( 'Client token must be a string' );
		}

		$this->client_token = $value;
	}


	/**
	 * Sets the response's nonce.
	 *
	 * This is necessary because First Data doesn't send it along in the response data, but rather a header.
	 *
	 * @since 4.7.0
	 *
	 * @param string $value token value
	 * @throws Framework\SV_WC_API_Exception
	 */
	public function set_nonce( $value ) {

		if ( ! is_string( $value ) ) {
			throw new Framework\SV_WC_API_Exception( 'Nonce must be a string' );
		}

		$this->nonce = $value;
	}


	/** Conditional methods *******************************************************************************************/


	/**
	 * Determines if there are errors in the response.
	 *
	 * @since 4.7.0
	 *
	 * @return bool
	 */
	public function has_errors() {

		return ! empty( $this->get_error() );
	}


	/** Utility methods ***********************************************************************************************/


	/**
	 * Converts the response data to a string that's safe for logging.
	 *
	 * Removes/masks any sensitive info.
	 *
	 * @since 4.7.0
	 *
	 * @return string
	 */
	public function to_string_safe() {

		return str_replace( $this->get_public_key(), str_repeat( '*', strlen( $this->get_public_key() ) ), $this->to_string() );
	}


}
