<?php
/*
Plugin Name: WPC Product Timer for WooCommerce
Plugin URI: https://wpclever.net/
Description: WPC Product Timer helps you add many actions for the product based on the conditionals of the time.
Version: 3.2.2
Author: WPClever
Author URI: https://wpclever.net
Text Domain: wpc-product-timer
Domain Path: /languages/
Requires at least: 4.0
Tested up to: 5.7.1
WC requires at least: 3.0
WC tested up to: 5.2.1
*/

defined( 'ABSPATH' ) || exit;

! defined( 'WOOPT_VERSION' ) && define( 'WOOPT_VERSION', '3.2.2' );
! defined( 'WOOPT_URI' ) && define( 'WOOPT_URI', plugin_dir_url( __FILE__ ) );
! defined( 'WOOPT_DOCS' ) && define( 'WOOPT_DOCS', 'http://doc.wpclever.net/woopt/' );
! defined( 'WOOPT_SUPPORT' ) && define( 'WOOPT_SUPPORT', 'https://wpclever.net/support?utm_source=support&utm_medium=woopt&utm_campaign=wporg' );
! defined( 'WOOPT_REVIEWS' ) && define( 'WOOPT_REVIEWS', 'https://wordpress.org/support/plugin/woo-product-timer/reviews/?filter=5' );
! defined( 'WOOPT_CHANGELOG' ) && define( 'WOOPT_CHANGELOG', 'https://wordpress.org/plugins/woo-product-timer/#developers' );
! defined( 'WOOPT_DISCUSSION' ) && define( 'WOOPT_DISCUSSION', 'https://wordpress.org/support/plugin/woo-product-timer' );
! defined( 'WPC_URI' ) && define( 'WPC_URI', WOOPT_URI );

include 'includes/wpc-dashboard.php';
include 'includes/wpc-menu.php';
include 'includes/wpc-kit.php';

if ( ! function_exists( 'woopt_init' ) ) {
	add_action( 'plugins_loaded', 'woopt_init', 11 );

	function woopt_init() {
		// load text-domain
		load_plugin_textdomain( 'wpc-product-timer', false, basename( __DIR__ ) . '/languages/' );

		if ( ! function_exists( 'WC' ) || ! version_compare( WC()->version, '3.0', '>=' ) ) {
			add_action( 'admin_notices', 'woopt_notice_wc' );

			return;
		}

		if ( ! class_exists( 'WPCleverWoopt' ) && class_exists( 'WC_Product' ) ) {
			class WPCleverWoopt {
				protected static $global_actions = array();

				function __construct() {
					self::$global_actions = get_option( 'woopt_actions' );

					// Menu
					add_action( 'admin_menu', array( $this, 'woopt_admin_menu' ) );

					// Enqueue backend scripts
					add_action( 'admin_enqueue_scripts', array( $this, 'woopt_admin_enqueue_scripts' ) );

					// Product data tabs
					add_filter( 'woocommerce_product_data_tabs', array( $this, 'woopt_product_data_tabs' ), 10, 1 );

					// Product data panels
					add_action( 'woocommerce_product_data_panels', array( $this, 'woopt_product_data_panels' ) );
					add_action( 'woocommerce_process_product_meta', array( $this, 'woopt_save_option_field' ) );

					// Add settings link
					add_filter( 'plugin_action_links', array( $this, 'woopt_action_links' ), 10, 2 );
					add_filter( 'plugin_row_meta', array( $this, 'woopt_row_meta' ), 10, 2 );

					// AJAX
					add_action( 'wp_ajax_woopt_add_conditional', array( $this, 'woopt_add_conditional' ) );

					// Features
					$features = get_option( 'woopt_features' );

					if ( empty( $features ) || in_array( 'stock', $features, false ) ) {
						add_filter( 'woocommerce_product_is_in_stock', array( $this, 'woopt_is_in_stock' ), 99, 2 );
					}

					if ( empty( $features ) || in_array( 'visibility', $features, false ) ) {
						add_filter( 'woocommerce_product_is_visible', array( $this, 'woopt_is_visible' ), 99, 2 );
						add_filter( 'woocommerce_variation_is_visible', array( $this, 'woopt_is_visible' ), 99, 2 );
						add_filter( 'woocommerce_variation_is_active', array( $this, 'woopt_is_visible' ), 99, 2 );
					}

					if ( empty( $features ) || in_array( 'featured', $features, false ) ) {
						add_filter( 'woocommerce_product_get_featured', array( $this, 'woopt_is_featured' ), 99, 2 );
					}

					if ( empty( $features ) || in_array( 'purchasable', $features, false ) ) {
						add_filter( 'woocommerce_is_purchasable', array( $this, 'woopt_is_purchasable' ), 99, 2 );
					}

					if ( empty( $features ) || in_array( 'individually', $features, false ) ) {
						add_filter( 'woocommerce_is_sold_individually', array(
							$this,
							'woopt_sold_individually'
						), 99, 2 );
					}

					if ( empty( $features ) || in_array( 'price', $features, false ) ) {
						add_filter( 'woocommerce_product_get_regular_price', array(
							$this,
							'woopt_get_regular_price'
						), 99, 2 );
						add_filter( 'woocommerce_product_get_sale_price', array(
							$this,
							'woopt_get_sale_price'
						), 99, 2 );
						add_filter( 'woocommerce_product_get_price', array( $this, 'woopt_get_price' ), 99, 2 );

						// Variation
						add_filter( 'woocommerce_product_variation_get_regular_price', array(
							$this,
							'woopt_get_regular_price'
						), 99, 2 );
						add_filter( 'woocommerce_product_variation_get_sale_price', array(
							$this,
							'woopt_get_sale_price'
						), 99, 2 );
						add_filter( 'woocommerce_product_variation_get_price', array(
							$this,
							'woopt_get_price'
						), 99, 2 );

						// Variations
						add_filter( 'woocommerce_variation_prices_regular_price', array(
							$this,
							'woopt_get_regular_price'
						), 99, 2 );
						add_filter( 'woocommerce_variation_prices_sale_price', array(
							$this,
							'woopt_get_sale_price'
						), 99, 2 );
						add_filter( 'woocommerce_variation_prices_price', array( $this, 'woopt_get_price' ), 99, 2 );
					}

					// Product columns
					add_filter( 'manage_edit-product_columns', array( $this, 'woopt_product_columns' ), 10 );
					add_action( 'manage_product_posts_custom_column', array(
						$this,
						'woopt_product_posts_custom_column'
					), 10, 2 );

					// Product class
					add_filter( 'woocommerce_post_class', array( $this, 'woopt_post_class' ), 99, 2 );

					// Export
					add_filter( 'woocommerce_product_export_column_names', array( $this, 'woopt_add_export_column' ) );
					add_filter( 'woocommerce_product_export_product_default_columns', array(
						$this,
						'woopt_add_export_column'
					) );
					add_filter( 'woocommerce_product_export_product_column_woopt_actions', array(
						$this,
						'woopt_add_export_data'
					), 10, 2 );

					// Import
					add_filter( 'woocommerce_csv_product_import_mapping_options', array(
						$this,
						'woopt_add_column_to_importer'
					) );
					add_filter( 'woocommerce_csv_product_import_mapping_default_columns', array(
						$this,
						'woopt_add_column_to_mapping_screen'
					) );
					add_filter( 'woocommerce_product_import_pre_insert_product_object', array(
						$this,
						'woopt_process_import'
					), 10, 2 );
				}

				public static function woopt_action_data( $action ) {
					$action_arr = explode( '|', $action );

					if ( strpos( $action, 'apply_' ) === false ) {
						// product
						return array(
							'action'      => isset( $action_arr[0] ) ? $action_arr[0] : '',
							'value'       => isset( $action_arr[1] ) ? $action_arr[1] : '100%',
							'conditional' => isset( $action_arr[2] ) ? $action_arr[2] : '',
						);
					} else {
						// global
						return array(
							'apply'       => isset( $action_arr[0] ) ? $action_arr[0] : 'apply_all',
							'apply_val'   => isset( $action_arr[1] ) ? $action_arr[1] : '',
							'action'      => isset( $action_arr[2] ) ? $action_arr[2] : '',
							'value'       => isset( $action_arr[3] ) ? $action_arr[3] : '100%',
							'conditional' => isset( $action_arr[4] ) ? $action_arr[4] : '',
						);
					}
				}

				public static function woopt_check_apply( $product, $apply, $apply_val ) {
					if ( is_numeric( $product ) ) {
						$product_id = $product;
					} else {
						$product_id = $product->get_id();
					}

					if ( ! $product_id ) {
						return false;
					}

					switch ( $apply ) {
						case 'apply_all':
							return true;
							break;
						case 'apply_product':
							if ( ! empty( $apply_val ) ) {
								$selected_ids = explode( ',', $apply_val );

								if ( in_array( $product_id, $selected_ids ) ) {
									return true;
								}
							}

							return false;
							break;
						case 'apply_category':
							if ( ! empty( $apply_val ) ) {
								$selected_cats = explode( ',', $apply_val );

								if ( has_term( $selected_cats, 'product_cat', $product_id ) ) {
									return true;
								}
							}

							return false;
							break;
						case 'apply_tag':
							if ( ! empty( $apply_val ) ) {
								$selected_tags = explode( ',', $apply_val );

								if ( has_term( $selected_tags, 'product_tag', $product_id ) ) {
									return true;
								}
							}

							return false;
							break;
						default:
							return false;
							break;
					}
				}

				public static function woopt_check_conditional( $conditionals ) {
					$condition        = true;
					$conditionals_arr = explode( '&', $conditionals );

					if ( count( $conditionals_arr ) > 0 ) {
						foreach ( $conditionals_arr as $conditional ) {
							$condition_item  = false;
							$conditional_arr = explode( '>', $conditional );

							if ( count( $conditional_arr ) > 0 ) {
								$conditional_key   = isset( $conditional_arr[0] ) ? $conditional_arr[0] : '';
								$conditional_value = isset( $conditional_arr[1] ) ? $conditional_arr[1] : '';

								if ( $conditional_value !== '' ) {
									switch ( $conditional_key ) {
										case 'date_range':
											$date_range_arr = explode( '-', $conditional_value );

											if ( count( $date_range_arr ) === 2 ) {
												$date_range_start = trim( $date_range_arr[0] );
												$date_range_end   = trim( $date_range_arr[1] );
												$current_time     = strtotime( current_time( 'm/d/Y' ) );

												if ( $current_time >= strtotime( $date_range_start ) && $current_time <= strtotime( $date_range_end ) ) {
													$condition_item = true;
												}
											} elseif ( count( $date_range_arr ) === 1 ) {
												$date_range_start = trim( $date_range_arr[0] );

												if ( strtotime( current_time( 'm/d/Y' ) ) === strtotime( $date_range_start ) ) {
													$condition_item = true;
												}
											}

											break;
										case 'date_multi':
											$multiple_dates_arr = explode( ', ', $conditional_value );

											if ( in_array( current_time( 'm/d/Y' ), $multiple_dates_arr, false ) ) {
												$condition_item = true;
											}

											break;
										case 'date_even':
											if ( (int) current_time( 'd' ) % 2 === 0 ) {
												$condition_item = true;
											}

											break;
										case 'date_odd':
											if ( (int) current_time( 'd' ) % 2 !== 0 ) {
												$condition_item = true;
											}

											break;
										case 'date_on':
											if ( strtotime( current_time( 'm/d/Y' ) ) === strtotime( $conditional_value ) ) {
												$condition_item = true;
											}

											break;
										case 'date_before':
											if ( strtotime( current_time( 'm/d/Y' ) ) < strtotime( $conditional_value ) ) {
												$condition_item = true;
											}

											break;
										case 'date_after':
											if ( strtotime( current_time( 'm/d/Y' ) ) > strtotime( $conditional_value ) ) {
												$condition_item = true;
											}

											break;
										case 'date_time_before':
											if ( current_time( 'timestamp' ) < strtotime( $conditional_value ) ) {
												$condition_item = true;
											}

											break;
										case 'date_time_after':
											if ( current_time( 'timestamp' ) > strtotime( $conditional_value ) ) {
												$condition_item = true;
											}

											break;
										case 'time_range':
											$time_range_arr = explode( '-', $conditional_value );

											if ( count( $time_range_arr ) === 2 ) {
												$current_time     = current_time( 'timestamp' );
												$current_date     = current_time( 'm/d/Y' );
												$time_range_start = $current_date . ' ' . trim( $time_range_arr[0] );
												$time_range_end   = $current_date . ' ' . trim( $time_range_arr[1] );

												if ( $current_time >= strtotime( $time_range_start ) && $current_time <= strtotime( $time_range_end ) ) {
													$condition_item = true;
												}
											}

											break;
										case 'time_before':
											$current_date = current_time( 'm/d/Y' );

											if ( current_time( 'timestamp' ) < strtotime( $current_date . ' ' . $conditional_value ) ) {
												$condition_item = true;
											}

											break;
										case 'time_after':
											$current_date = current_time( 'm/d/Y' );

											if ( current_time( 'timestamp' ) > strtotime( $current_date . ' ' . $conditional_value ) ) {
												$condition_item = true;
											}

											break;
										case 'weekly_every':
											if ( strtolower( current_time( 'D' ) ) === $conditional_value ) {
												$condition_item = true;
											}

											break;
										case 'week_no':
											if ( (int) current_time( 'W' ) === (int) $conditional_value ) {
												$condition_item = true;
											}

											break;
										case 'every_day':
											$condition_item = true;

											break;
									}
								}
							}

							$condition &= $condition_item;
						}
					}

					return $condition;
				}

				public static function woopt_get_action_result( $result, $product, $action_true = '', $action_false = '' ) {
					$variation_id = 0;
					$product_id   = $product->get_id();

					if ( $product->is_type( 'variation' ) ) {
						$variation_id = $product_id;
						$product_id   = $product->get_parent_id();
					}

					// global actions
					if ( is_array( self::$global_actions ) && ( count( self::$global_actions ) > 0 ) ) {
						foreach ( self::$global_actions as $global_action ) {
							$action_data        = self::woopt_action_data( $global_action );
							$action_apply       = $action_data['apply'];
							$action_apply_val   = $action_data['apply_val'];
							$action_key         = $action_data['action'];
							$action_conditional = $action_data['conditional'];

							if ( $action_key !== $action_true && $action_key !== $action_false ) {
								continue;
							}

							if ( self::woopt_check_apply( $product_id, $action_apply, $action_apply_val ) && self::woopt_check_conditional( $action_conditional ) ) {
								if ( $action_key === $action_true ) {
									$result = true;
								} else {
									$result = false;
								}
							}
						}
					}

					// product actions
					$actions = get_post_meta( $product_id, 'woopt_actions', true );

					if ( is_array( $actions ) && ( count( $actions ) > 0 ) ) {
						foreach ( $actions as $action ) {
							$action_data        = self::woopt_action_data( $action );
							$action_key         = $action_data['action'];
							$action_conditional = $action_data['conditional'];

							if ( $action_key !== $action_true && $action_key !== $action_false ) {
								continue;
							}

							if ( self::woopt_check_conditional( $action_conditional ) ) {
								if ( $action_key === $action_true ) {
									$result = true;
								} else {
									$result = false;
								}
							}
						}
					}

					// global actions for variation
					if ( $variation_id && is_array( self::$global_actions ) && ( count( self::$global_actions ) > 0 ) ) {
						foreach ( self::$global_actions as $global_action ) {
							$action_data        = self::woopt_action_data( $global_action );
							$action_apply       = $action_data['apply'];
							$action_apply_val   = $action_data['apply_val'];
							$action_key         = $action_data['action'];
							$action_conditional = $action_data['conditional'];

							if ( $action_apply !== 'apply_product' ) {
								continue;
							}

							if ( $action_key !== $action_true && $action_key !== $action_false ) {
								continue;
							}

							if ( self::woopt_check_apply( $variation_id, $action_apply, $action_apply_val ) && self::woopt_check_conditional( $action_conditional ) ) {
								if ( $action_key === $action_true ) {
									$result = true;
								} else {
									$result = false;
								}
							}
						}
					}

					return $result;
				}

				public static function woopt_post_class( $classes, $product ) {
					if ( $product && $product->is_type( 'variation' ) && $product->get_parent_id() ) {
						$product_id = $product->get_parent_id();
					} else {
						$product_id = $product->get_id();
					}

					// global actions
					if ( is_array( self::$global_actions ) && ( count( self::$global_actions ) > 0 ) ) {
						foreach ( self::$global_actions as $global_action ) {
							$action_data        = self::woopt_action_data( $global_action );
							$action_apply       = $action_data['apply'];
							$action_apply_val   = $action_data['apply_val'];
							$action_key         = $action_data['action'];
							$action_conditional = $action_data['conditional'];

							if ( self::woopt_check_apply( $product_id, $action_apply, $action_apply_val ) && self::woopt_check_conditional( $action_conditional ) ) {
								$classes[] = 'woopt_global';
								$classes[] = 'woopt_global_' . $action_key;
							}
						}
					}

					$actions = get_post_meta( $product_id, 'woopt_actions', true );

					if ( is_array( $actions ) && ( count( $actions ) > 0 ) ) {
						foreach ( $actions as $action ) {
							$action_data        = self::woopt_action_data( $action );
							$action_key         = $action_data['action'];
							$action_conditional = $action_data['conditional'];

							if ( self::woopt_check_conditional( $action_conditional ) ) {
								$classes[] = 'woopt';
								$classes[] = 'woopt_' . $action_key;
							}
						}
					}

					return $classes;
				}

				public static function woopt_get_regular_price( $regular_price, $product ) {
					$variation_id = 0;
					$product_id   = $product->get_id();

					if ( $product->is_type( 'variation' ) ) {
						$variation_id = $product_id;
						$product_id   = $product->get_parent_id();
					}

					// global actions
					if ( is_array( self::$global_actions ) && ( count( self::$global_actions ) > 0 ) ) {
						foreach ( self::$global_actions as $global_action ) {
							$action_data        = self::woopt_action_data( $global_action );
							$action_apply       = $action_data['apply'];
							$action_apply_val   = $action_data['apply_val'];
							$action_key         = $action_data['action'];
							$action_price       = $action_data['value'];
							$action_conditional = $action_data['conditional'];

							if ( ( $action_key === 'set_regularprice' ) && self::woopt_check_apply( $product_id, $action_apply, $action_apply_val ) && self::woopt_check_conditional( $action_conditional ) ) {
								if ( strpos( $action_price, '%' ) !== false ) {
									$percentage   = (float) preg_replace( '/[^0-9.]/', '', $action_price );
									$action_price = (float) $regular_price * $percentage / 100;
								}

								$regular_price = $action_price;
							}
						}
					}

					// product actions
					$actions = get_post_meta( $product_id, 'woopt_actions', true );

					if ( is_array( $actions ) && ( count( $actions ) > 0 ) ) {
						foreach ( $actions as $action ) {
							$action_data        = self::woopt_action_data( $action );
							$action_key         = $action_data['action'];
							$action_price       = $action_data['value'];
							$action_conditional = $action_data['conditional'];

							if ( ( $action_key === 'set_regularprice' ) && self::woopt_check_conditional( $action_conditional ) ) {
								if ( strpos( $action_price, '%' ) !== false ) {
									$percentage   = (float) preg_replace( '/[^0-9.]/', '', $action_price );
									$action_price = (float) $regular_price * $percentage / 100;
								}

								$regular_price = $action_price;
							}
						}
					}

					// global actions for variation
					if ( $variation_id && is_array( self::$global_actions ) && ( count( self::$global_actions ) > 0 ) ) {
						foreach ( self::$global_actions as $global_action ) {
							$action_data        = self::woopt_action_data( $global_action );
							$action_apply       = $action_data['apply'];
							$action_apply_val   = $action_data['apply_val'];
							$action_key         = $action_data['action'];
							$action_price       = $action_data['value'];
							$action_conditional = $action_data['conditional'];

							if ( $action_apply !== 'apply_product' ) {
								continue;
							}

							if ( ( $action_key === 'set_regularprice' ) && self::woopt_check_apply( $variation_id, $action_apply, $action_apply_val ) && self::woopt_check_conditional( $action_conditional ) ) {
								if ( strpos( $action_price, '%' ) !== false ) {
									$percentage   = (float) preg_replace( '/[^0-9.]/', '', $action_price );
									$action_price = (float) $regular_price * $percentage / 100;
								}

								$regular_price = $action_price;
							}
						}
					}

					return apply_filters( 'woopt_get_regular_price', $regular_price, $product );
				}

				public static function woopt_get_sale_price( $sale_price, $product ) {
					$variation_id = 0;
					$product_id   = $product->get_id();

					if ( $product->is_type( 'variation' ) ) {
						$variation_id = $product_id;
						$product_id   = $product->get_parent_id();
					}

					// global actions
					if ( is_array( self::$global_actions ) && ( count( self::$global_actions ) > 0 ) ) {
						foreach ( self::$global_actions as $global_action ) {
							$action_data        = self::woopt_action_data( $global_action );
							$action_apply       = $action_data['apply'];
							$action_apply_val   = $action_data['apply_val'];
							$action_key         = $action_data['action'];
							$action_price       = $action_data['value'];
							$action_conditional = $action_data['conditional'];

							if ( ( $action_key === 'set_saleprice' ) && self::woopt_check_apply( $product_id, $action_apply, $action_apply_val ) && self::woopt_check_conditional( $action_conditional ) ) {
								if ( strpos( $action_price, '%' ) !== false ) {
									$percentage   = (float) preg_replace( '/[^0-9.]/', '', $action_price );
									$action_price = (float) $sale_price * $percentage / 100;
								}

								$sale_price = $action_price;
							}
						}
					}

					// product actions
					$actions = get_post_meta( $product_id, 'woopt_actions', true );

					if ( is_array( $actions ) && ( count( $actions ) > 0 ) ) {
						foreach ( $actions as $action ) {
							$action_data        = self::woopt_action_data( $action );
							$action_key         = $action_data['action'];
							$action_price       = $action_data['value'];
							$action_conditional = $action_data['conditional'];

							if ( ( $action_key === 'set_saleprice' ) && self::woopt_check_conditional( $action_conditional ) ) {
								if ( strpos( $action_price, '%' ) !== false ) {
									$percentage   = (float) preg_replace( '/[^0-9.]/', '', $action_price );
									$action_price = (float) $sale_price * $percentage / 100;
								}

								$sale_price = $action_price;
							}
						}
					}

					// global actions for variation
					if ( $variation_id && is_array( self::$global_actions ) && ( count( self::$global_actions ) > 0 ) ) {
						foreach ( self::$global_actions as $global_action ) {
							$action_data        = self::woopt_action_data( $global_action );
							$action_apply       = $action_data['apply'];
							$action_apply_val   = $action_data['apply_val'];
							$action_key         = $action_data['action'];
							$action_price       = $action_data['value'];
							$action_conditional = $action_data['conditional'];

							if ( $action_apply !== 'apply_product' ) {
								continue;
							}

							if ( ( $action_key === 'set_saleprice' ) && self::woopt_check_apply( $variation_id, $action_apply, $action_apply_val ) && self::woopt_check_conditional( $action_conditional ) ) {
								if ( strpos( $action_price, '%' ) !== false ) {
									$percentage   = (float) preg_replace( '/[^0-9.]/', '', $action_price );
									$action_price = (float) $sale_price * $percentage / 100;
								}

								$sale_price = $action_price;
							}
						}
					}

					return apply_filters( 'woopt_get_sale_price', $sale_price, $product );
				}

				public static function woopt_get_price( $price, $product ) {
					if ( $product->is_on_sale() ) {
						return self::woopt_get_sale_price( $price, $product );
					}

					return self::woopt_get_regular_price( $price, $product );
				}

				public static function woopt_is_in_stock( $in_stock, $product ) {
					$in_stock = self::woopt_get_action_result( $in_stock, $product, 'set_instock', 'set_outofstock' );

					return apply_filters( 'woopt_is_in_stock', $in_stock, $product );
				}

				public static function woopt_is_visible( $visible, $product_id ) {
					$_product = wc_get_product( $product_id );
					$visible  = self::woopt_get_action_result( $visible, $_product, 'set_visible', 'set_hidden' );

					return apply_filters( 'woopt_is_visible', $visible, $_product );
				}

				public static function woopt_is_featured( $featured, $product ) {
					$featured = self::woopt_get_action_result( $featured, $product, 'set_featured', 'set_unfeatured' );

					return apply_filters( 'woopt_is_featured', $featured, $product );
				}

				public static function woopt_is_purchasable( $purchasable, $product ) {
					$purchasable = self::woopt_get_action_result( $purchasable, $product, 'set_purchasable', 'set_unpurchasable' );

					return apply_filters( 'woopt_is_purchasable', $purchasable, $product );
				}

				public static function woopt_sold_individually( $sold_individually, $product ) {
					$sold_individually = self::woopt_get_action_result( $sold_individually, $product, 'enable_sold_individually', 'disable_sold_individually' );

					return apply_filters( 'woopt_sold_individually', $sold_individually, $product );
				}

				function woopt_admin_menu() {
					add_submenu_page( 'wpclever', esc_html__( 'WPC Product Timer', 'wpc-product-timer' ), esc_html__( 'Product Timer', 'wpc-product-timer' ), 'manage_options', 'wpclever-woopt', array(
						&$this,
						'woopt_admin_menu_content'
					) );
				}

				function woopt_admin_menu_content() {
					$active_tab = isset( $_GET['tab'] ) ? sanitize_key( $_GET['tab'] ) : 'global';
					?>
                    <div class="wpclever_settings_page wrap">
                        <h1 class="wpclever_settings_page_title"><?php echo esc_html__( 'WPC Product Timer', 'wpc-product-timer' ) . ' ' . WOOPT_VERSION; ?></h1>
                        <div class="wpclever_settings_page_desc about-text">
                            <p>
								<?php printf( esc_html__( 'Thank you for using our plugin! If you are satisfied, please reward it a full five-star %s rating.', 'wpc-product-timer' ), '<span style="color:#ffb900">&#9733;&#9733;&#9733;&#9733;&#9733;</span>' ); ?>
                                <br/>
                                <a href="<?php echo esc_url( WOOPT_REVIEWS ); ?>"
                                   target="_blank"><?php esc_html_e( 'Reviews', 'wpc-product-timer' ); ?></a> | <a
                                        href="<?php echo esc_url( WOOPT_CHANGELOG ); ?>"
                                        target="_blank"><?php esc_html_e( 'Changelog', 'wpc-product-timer' ); ?></a>
                                | <a href="<?php echo esc_url( WOOPT_DISCUSSION ); ?>"
                                     target="_blank"><?php esc_html_e( 'Discussion', 'wpc-product-timer' ); ?></a>
                            </p>
                        </div>
                        <div class="wpclever_settings_page_nav">
                            <h2 class="nav-tab-wrapper">
                                <a href="<?php echo admin_url( 'admin.php?page=wpclever-woopt&tab=how' ); ?>"
                                   class="<?php echo $active_tab === 'how' ? 'nav-tab nav-tab-active' : 'nav-tab'; ?>">
									<?php esc_html_e( 'How to use?', 'wpc-product-timer' ); ?>
                                </a>
                                <a href="<?php echo admin_url( 'admin.php?page=wpclever-woopt&tab=global' ); ?>"
                                   class="<?php echo $active_tab === 'global' ? 'nav-tab nav-tab-active' : 'nav-tab'; ?>">
									<?php esc_html_e( 'Global Timer', 'wpc-product-timer' ); ?>
                                </a>
                                <a href="<?php echo esc_url( WOOPT_DOCS ); ?>" class="nav-tab" target="_blank">
									<?php esc_html_e( 'Docs', 'wpc-product-timer' ); ?>
                                </a>
                                <a href="<?php echo admin_url( 'admin.php?page=wpclever-woopt&tab=premium' ); ?>"
                                   class="<?php echo $active_tab === 'premium' ? 'nav-tab nav-tab-active' : 'nav-tab'; ?>"
                                   style="color: #c9356e">
									<?php esc_html_e( 'Premium Version', 'wpc-product-timer' ); ?>
                                </a>
                                <a href="<?php echo esc_url( admin_url( 'admin.php?page=wpclever-kit' ) ); ?>"
                                   class="nav-tab">
									<?php esc_html_e( 'Essential Kit', 'wpc-product-timer' ); ?>
                                </a>
                            </h2>
                        </div>
                        <div class="wpclever_settings_page_content">
							<?php if ( $active_tab === 'how' ) { ?>
                                <div class="wpclever_settings_page_content_text">
                                    <p>
										<?php esc_html_e( '1. Global timer: Switch to Global Timer tab to set the timer for all products, categories or tags.', 'wpc-product-timer' ); ?>
                                    </p>
                                    <p>
										<?php esc_html_e( '2. Product basis timer: When adding/editing the product you can choose the Timer tab then add action & time conditional.', 'wpc-product-timer' ); ?>
                                    </p>
                                </div>
							<?php } elseif ( $active_tab === 'global' ) { ?>
                                <form method="post" action="options.php">
									<?php wp_nonce_field( 'update-options' ) ?>
                                    <table class="form-table">
                                        <tr>
                                            <th>
												<?php esc_html_e( 'Performance', 'wpc-product-timer' ); ?>
                                            </th>
                                            <td>
												<?php $features = get_option( 'woopt_features' ); ?>
                                                <ul>
                                                    <li>
                                                        <input type="checkbox" name="woopt_features[]"
                                                               value="stock" <?php echo( empty( $features ) || in_array( 'stock', $features, false ) ? 'checked' : '' ); ?>/>
														<?php esc_html_e( 'Stock (in stock, out of stock)', 'wpc-product-timer' ); ?>
                                                    </li>
                                                    <li>
                                                        <input type="checkbox" name="woopt_features[]"
                                                               value="visibility" <?php echo( empty( $features ) || in_array( 'visibility', $features, false ) ? 'checked' : '' ); ?>/>
														<?php esc_html_e( 'Visibility (visible, hidden)', 'wpc-product-timer' ); ?>
                                                    </li>
                                                    <li>
                                                        <input type="checkbox" name="woopt_features[]"
                                                               value="featured" <?php echo( empty( $features ) || in_array( 'featured', $features, false ) ? 'checked' : '' ); ?>/>
														<?php esc_html_e( 'Featured (featured, unfeatured)', 'wpc-product-timer' ); ?>
                                                    </li>
                                                    <li>
                                                        <input type="checkbox" name="woopt_features[]"
                                                               value="purchasable" <?php echo( empty( $features ) || in_array( 'purchasable', $features, false ) ? 'checked' : '' ); ?>/>
														<?php esc_html_e( 'Purchasable (purchasable, unpurchasable)', 'wpc-product-timer' ); ?>
                                                    </li>
                                                    <li>
                                                        <input type="checkbox" name="woopt_features[]"
                                                               value="price" <?php echo( empty( $features ) || in_array( 'price', $features, false ) ? 'checked' : '' ); ?>/>
														<?php esc_html_e( 'Price (regular price, sale price)', 'wpc-product-timer' ); ?>
                                                    </li>
                                                    <li>
                                                        <input type="checkbox" name="woopt_features[]"
                                                               value="individually" <?php echo( empty( $features ) || in_array( 'individually', $features, false ) ? 'checked' : '' ); ?>/>
														<?php esc_html_e( 'Sold individually (enable, disable)', 'wpc-product-timer' ); ?>
                                                    </li>
                                                </ul>
                                                <span class="description"><?php esc_html_e( 'Uncheck the feature(s) you don\'t use in all timers for better performance.', 'wpc-product-timer' ); ?></span>
                                            </td>
                                        </tr>
                                        <tr>
                                            <th>
												<?php esc_html_e( 'Current time', 'wpc-product-timer' ); ?>
                                            </th>
                                            <td>
                                                <code><?php echo current_time( 'l' ); ?></code>
                                                <code><?php echo current_time( 'm/d/Y' ); ?></code>
                                                <code><?php echo current_time( 'h:i a' ); ?></code>
                                                <code><?php echo esc_html__( 'Week No.', 'wpc-product-timer' ) . ' ' . current_time( 'W' ); ?></code>
                                                <a
                                                        href="<?php echo admin_url( 'options-general.php' ); ?>"
                                                        target="_blank"><?php esc_html_e( 'Date/time settings', 'wpc-product-timer' ); ?></a>
                                            </td>
                                        </tr>
                                        <tr>
                                            <th>
												<?php esc_html_e( 'Actions', 'wpc-product-timer' ); ?>
                                            </th>
                                            <td>
                                                <div class="woopt_actions">
													<?php
													if ( is_array( self::$global_actions ) && ( count( self::$global_actions ) > 0 ) ) {
														foreach ( self::$global_actions as $action ) {
															self::woopt_action( $action, true );
														}
													} else {
														self::woopt_action( null, true );
													}
													?>
                                                </div>
                                                <div class="woopt_add_action">
                                                    <a href="https://wpclever.net/downloads/woocommerce-product-timer?utm_source=pro&utm_medium=woopt&utm_campaign=wporg"
                                                       target="_blank"
                                                       onclick="return confirm('This feature only available in Premium Version!\nBuy it now? Just $29')">
														<?php esc_html_e( '+ Add action', 'wpc-product-timer' ); ?>
                                                    </a>
                                                </div>
                                            </td>
                                        </tr>
                                        <tr class="submit">
                                            <th colspan="2">
                                                <input type="submit" name="submit" class="button button-primary"
                                                       value="<?php esc_html_e( 'Update Options', 'wpc-product-timer' ); ?>"/>
                                                <input type="hidden" name="action" value="update"/>
                                                <input type="hidden" name="page_options"
                                                       value="woopt_features,woopt_actions"/>
                                            </th>
                                        </tr>
                                    </table>
                                </form>
							<?php } elseif ( $active_tab === 'premium' ) { ?>
                                <div class="wpclever_settings_page_content_text">
                                    <p>Get the Premium Version just $29! <a
                                                href="https://wpclever.net/downloads/woocommerce-product-timer?utm_source=pro&utm_medium=woopt&utm_campaign=wporg"
                                                target="_blank">https://wpclever.net/downloads/woocommerce-product-timer</a>
                                    </p>
                                    <p><strong>Extra features for Premium Version:</strong></p>
                                    <ul style="margin-bottom: 0">
                                        <li>- Add multiple actions for one product.</li>
                                        <li>- Get the lifetime update & premium support.</li>
                                    </ul>
                                </div>
							<?php } ?>
                        </div>
                    </div>
					<?php
				}

				function woopt_admin_enqueue_scripts() {
					wp_enqueue_style( 'wooptdpk', WOOPT_URI . 'assets/libs/wooptdpk/css/datepicker.css' );
					wp_enqueue_script( 'wooptdpk', WOOPT_URI . 'assets/libs/wooptdpk/js/datepicker.js', array( 'jquery' ), WOOPT_VERSION, true );

					wp_enqueue_style( 'select2', WOOPT_URI . 'assets/css/select2.min.css' );

					wp_enqueue_style( 'woopt-backend', WOOPT_URI . 'assets/css/backend.css' );
					wp_enqueue_script( 'woopt-backend', WOOPT_URI . 'assets/js/backend.js', array(
						'jquery',
						'wc-enhanced-select'
					), WOOPT_VERSION, true );
					wp_localize_script( 'woopt-backend', 'woopt_vars', array(
							'woopt_nonce' => wp_create_nonce( 'woopt_nonce' )
						)
					);
				}

				function woopt_action_links( $links, $file ) {
					static $plugin;

					if ( ! isset( $plugin ) ) {
						$plugin = plugin_basename( __FILE__ );
					}

					if ( $plugin === $file ) {
						$how              = '<a href="' . admin_url( 'admin.php?page=wpclever-woopt&tab=how' ) . '">' . esc_html__( 'How to use?', 'wpc-product-timer' ) . '</a>';
						$global           = '<a href="' . admin_url( 'admin.php?page=wpclever-woopt&tab=global' ) . '">' . esc_html__( 'Global Timer', 'wpc-product-timer' ) . '</a>';
						$links['premium'] = '<a href="' . admin_url( 'admin.php?page=wpclever-woopt&tab=premium' ) . '">' . esc_html__( 'Premium Version', 'wpc-product-timer' ) . '</a>';
						array_unshift( $links, $how, $global );
					}

					return (array) $links;
				}

				function woopt_row_meta( $links, $file ) {
					static $plugin;

					if ( ! isset( $plugin ) ) {
						$plugin = plugin_basename( __FILE__ );
					}

					if ( $plugin === $file ) {
						$row_meta = array(
							'docs'    => '<a href="' . esc_url( WOOPT_DOCS ) . '" target="_blank">' . esc_html__( 'Docs', 'wpc-product-timer' ) . '</a>',
							'support' => '<a href="' . esc_url( WOOPT_SUPPORT ) . '" target="_blank">' . esc_html__( 'Support', 'wpc-product-timer' ) . '</a>',
						);

						return array_merge( $links, $row_meta );
					}

					return (array) $links;
				}

				function woopt_product_data_tabs( $tabs ) {
					$tabs['woopt'] = array(
						'label'  => esc_html__( 'Timer', 'wpc-product-timer' ),
						'target' => 'woopt_settings',
					);

					return $tabs;
				}

				function woopt_product_data_panels() {
					global $post;
					$post_id = $post->ID;
					$actions = get_post_meta( $post_id, 'woopt_actions', true );
					?>
                    <div id='woopt_settings' class='panel woocommerce_options_panel woopt_settings'>
                        <div class="woopt_global_timer"><span
                                    class="dashicons dashicons-admin-site"></span> <a
                                    href="<?php echo admin_url( 'admin.php?page=wpclever-woopt&tab=global' ); ?>"
                                    target="_blank"><?php esc_html_e( 'Click here to configure the Global Timer', 'wpc-product-timer' ); ?></a>
                        </div>
                        <div class="woopt_current_time">
							<?php esc_html_e( 'Current time', 'wpc-product-timer' ); ?>
                            <code><?php echo current_time( 'l' ); ?></code>
                            <code><?php echo current_time( 'm/d/Y' ); ?></code>
                            <code><?php echo current_time( 'h:i a' ); ?></code>
                            <code><?php echo esc_html__( 'Week No.', 'wpc-product-timer' ) . ' ' . current_time( 'W' ); ?></code>
                            <a
                                    href="<?php echo admin_url( 'options-general.php' ); ?>"
                                    target="_blank"><?php esc_html_e( 'Date/time settings', 'wpc-product-timer' ); ?></a>
                        </div>
                        <div class="woopt_actions">
							<?php
							if ( is_array( $actions ) && ( count( $actions ) > 0 ) ) {
								foreach ( $actions as $action ) {
									self::woopt_action( $action );
								}
							} else {
								self::woopt_action();
							}
							?>
                        </div>
                        <div class="woopt_add_action">
                            <a href="https://wpclever.net/downloads/woocommerce-product-timer?utm_source=pro&utm_medium=woopt&utm_campaign=wporg"
                               target="_blank"
                               onclick="return confirm('This feature only available in Premium Version!\nBuy it now? Just $29')">
								<?php esc_html_e( '+ Add action', 'wpc-product-timer' ); ?>
                            </a>
                        </div>
                    </div>
					<?php
				}

				function woopt_action( $action = null, $global = false ) {
					$action_data = self::woopt_action_data( $action );
					$apply       = isset( $action_data['apply'] ) ? $action_data['apply'] : 'apply_all';
					$apply_val   = isset( $action_data['apply_val'] ) ? $action_data['apply_val'] : '';
					$action      = $action_data['action'];
					$price       = $action_data['value'];
					$conditional = $action_data['conditional'];
					?>
                    <div class="woopt_action">
                        <span class="woopt_action_remove">&times;</span>
						<?php if ( $global ) { ?>
                            <div class="woopt_tr">
                                <div class="woopt_th"><?php esc_html_e( 'Apply for', 'wpc-product-timer' ); ?></div>
                                <div class="woopt_td woopt_action_td">
                                    <select class="woopt_apply_selector">
                                        <option value="apply_all"><?php esc_html_e( 'All products', 'wpc-product-timer' ); ?></option>
                                        <option value="apply_product" <?php echo( $apply === 'apply_product' ? 'selected' : '' ); ?>>
											<?php esc_html_e( 'Products', 'wpc-product-timer' ); ?>
                                        </option>
                                        <option value="apply_category" <?php echo( $apply === 'apply_category' ? 'selected' : '' ); ?>>
											<?php esc_html_e( 'Categories', 'wpc-product-timer' ); ?>
                                        </option>
                                        <option value="apply_tag" <?php echo( $apply === 'apply_tag' ? 'selected' : '' ); ?>>
											<?php esc_html_e( 'Tags', 'wpc-product-timer' ); ?>
                                        </option>
                                    </select>
                                </div>
                            </div>
                            <div class="woopt_tr hide_apply show_if_apply_product" style="display: none">
                                <div class="woopt_th"><?php esc_html_e( 'Products', 'wpc-product-timer' ); ?></div>
                                <div class="woopt_td woopt_action_td">
                                    <select class="wc-product-search woopt-product-search" multiple="multiple"
                                            data-placeholder="<?php esc_attr_e( 'Search for a product&hellip;', 'wpc-product-timer' ); ?>"
                                            data-action="woocommerce_json_search_products_and_variations">
										<?php
										$_product_ids = explode( ',', $apply_val );

										foreach ( $_product_ids as $_product_id ) {
											$_product = wc_get_product( $_product_id );

											if ( $_product ) {
												echo '<option value="' . esc_attr( $_product_id ) . '" selected="selected">' . wp_kses_post( $_product->get_formatted_name() ) . '</option>';
											}
										}
										?>
                                    </select>
									<?php echo '<script>jQuery(document.body).trigger( \'wc-enhanced-select-init\' );</script>'; ?>
                                </div>
                            </div>
                            <div class="woopt_tr hide_apply show_if_apply_category" style="display: none">
                                <div class="woopt_th"><?php esc_html_e( 'Categories', 'wpc-product-timer' ); ?></div>
                                <div class="woopt_td woopt_action_td">
                                    <select class="wc-category-search woopt-category-search" multiple="multiple"
                                            data-placeholder="<?php esc_attr_e( 'Search for a category&hellip;', 'wpc-product-timer' ); ?>">
										<?php
										$category_slugs = explode( ',', $apply_val );

										if ( count( $category_slugs ) > 0 ) {
											foreach ( $category_slugs as $category_slug ) {
												$category = get_term_by( 'slug', $category_slug, 'product_cat' );

												if ( $category ) {
													echo '<option value="' . esc_attr( $category_slug ) . '" selected="selected">' . wp_kses_post( $category->name ) . '</option>';
												}
											}
										}
										?>
                                    </select>
									<?php echo '<script>jQuery(document.body).trigger( \'wc-enhanced-select-init\' );</script>'; ?>
                                </div>
                            </div>
                            <div class="woopt_tr hide_apply show_if_apply_tag" style="display: none">
                                <div class="woopt_th"><?php esc_html_e( 'Tags', 'wpc-product-timer' ); ?></div>
                                <div class="woopt_td woopt_action_td">
                                    <input class="woopt_apply_val" type="text" style="width: 100%"
                                           value="<?php echo esc_attr( $apply_val ); ?>"/>
                                    <span class="description"><?php esc_html_e( 'Add some tags, split by a comma.', 'wpc-product-timer' ); ?></span>
                                </div>
                            </div>
						<?php } ?>
                        <div class="woopt_tr">
                            <div class="woopt_th"><?php esc_html_e( 'Action', 'wpc-product-timer' ); ?></div>
                            <div class="woopt_td woopt_action_td">
                                <input class="woopt_action_val" type="hidden" name="woopt_actions[]" value=""/>
                                <span>
							<select class="woopt_action_selector">
								<option value=""><?php esc_html_e( 'Choose action', 'wpc-product-timer' ); ?></option>
								<option value="set_instock" <?php echo( $action === 'set_instock' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Set in stock', 'wpc-product-timer' ); ?>
								</option>
								<option value="set_outofstock" <?php echo( $action === 'set_outofstock' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Set out of stock', 'wpc-product-timer' ); ?>
								</option>
								<option value="set_visible" <?php echo( $action === 'set_visible' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Set visible', 'wpc-product-timer' ); ?>
								</option>
								<option value="set_hidden" <?php echo( $action === 'set_hidden' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Set hidden', 'wpc-product-timer' ); ?>
								</option>
                                <option value="set_featured" <?php echo( $action === 'set_featured' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Set featured', 'wpc-product-timer' ); ?>
								</option>
                                <option value="set_unfeatured" <?php echo( $action === 'set_unfeatured' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Set unfeatured', 'wpc-product-timer' ); ?>
								</option>
								<option value="set_purchasable" <?php echo( $action === 'set_purchasable' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Set purchasable', 'wpc-product-timer' ); ?>
								</option>
								<option value="set_unpurchasable" <?php echo( $action === 'set_unpurchasable' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Set unpurchasable', 'wpc-product-timer' ); ?>
								</option>
								<option value="set_regularprice"
                                        data-show="price" <?php echo( $action === 'set_regularprice' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Set regular price', 'wpc-product-timer' ); ?>
								</option>
								<option value="set_saleprice"
                                        data-show="price" <?php echo( $action === 'set_saleprice' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Set sale price', 'wpc-product-timer' ); ?>
								</option>
                                <option value="enable_sold_individually"  <?php echo( $action === 'enable_sold_individually' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Enable sold individually', 'wpc-product-timer' ); ?>
								</option>
                                <option value="disable_sold_individually"  <?php echo( $action === 'disable_sold_individually' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Disable sold individually', 'wpc-product-timer' ); ?>
								</option>
							</select>
						</span>
                                <span class="woopt_hide woopt_show_if_price">
							<input class="woopt_price"
                                   value="<?php echo $price; ?>"
                                   type="text" placeholder="price or percentage" style="width: 150px"/>
						</span>
                            </div>
                        </div>
                        <div class="woopt_tr">
                            <div class="woopt_th"><?php esc_html_e( 'Time conditional', 'wpc-product-timer' ); ?></div>
                            <div class="woopt_td">
                                <div class="woopt_conditionals">
									<?php self::woopt_conditional( $conditional ); ?>
                                </div>
                                <div class="woopt_add_conditional">
                                    <a href="#"
                                       class="woopt_new_conditional"><?php esc_html_e( '+ Add time', 'wpc-product-timer' ); ?></a>
                                </div>
                            </div>
                        </div>
                    </div>
					<?php
				}

				function woopt_conditional( $conditional = null ) {
					$conditional_arr = explode( '&', $conditional );

					if ( is_array( $conditional_arr ) && ( count( $conditional_arr ) > 0 ) ) {
						foreach ( $conditional_arr as $conditional_item ) {
							$conditional_item_arr       = explode( '>', $conditional_item );
							$conditional_item_key       = isset( $conditional_item_arr[0] ) ? $conditional_item_arr[0] : '';
							$conditional_item_val       = trim( isset( $conditional_item_arr[1] ) ? $conditional_item_arr[1] : '' );
							$conditional_item_time_from = '';
							$conditional_item_time_to   = '';

							if ( $conditional_item_key === 'time_range' ) {
								$conditional_item_time      = explode( '-', $conditional_item_val );
								$conditional_item_time_from = trim( isset( $conditional_item_time[0] ) ? $conditional_item_time[0] : '' );
								$conditional_item_time_to   = trim( isset( $conditional_item_time[1] ) ? $conditional_item_time[1] : '' );
							}
							?>
                            <div class="woopt_conditional_item">
                                <span class="woopt_conditional_remove">&times;</span>
                                <span>
							<select class="woopt_conditional">
								<option value=""><?php esc_html_e( 'Choose the time', 'wpc-product-timer' ); ?></option>
								<option value="date_on"
                                        data-show="date" <?php echo( $conditional_item_key === 'date_on' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'On the date', 'wpc-product-timer' ); ?>
								</option>
                                <option value="date_time_before"
                                        data-show="date_time" <?php echo( $conditional_item_key === 'date_time_before' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Before date & time', 'wpc-product-timer' ); ?>
								</option>
								<option value="date_time_after"
                                        data-show="date_time" <?php echo( $conditional_item_key === 'date_time_after' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'After date & time', 'wpc-product-timer' ); ?>
								</option>
								<option value="date_before"
                                        data-show="date" <?php echo( $conditional_item_key === 'date_before' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Before date', 'wpc-product-timer' ); ?>
								</option>
								<option value="date_after"
                                        data-show="date" <?php echo( $conditional_item_key === 'date_after' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'After date', 'wpc-product-timer' ); ?>
								</option>
								<option value="date_multi"
                                        data-show="date_multi" <?php echo( $conditional_item_key === 'date_multi' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Multiple dates', 'wpc-product-timer' ); ?>
								</option>
								<option value="date_range"
                                        data-show="date_range" <?php echo( $conditional_item_key === 'date_range' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Date range', 'wpc-product-timer' ); ?>
								</option>
								<option value="date_even"
                                        data-show="none" <?php echo( $conditional_item_key === 'date_even' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'All even dates', 'wpc-product-timer' ); ?>
								</option>
								<option value="date_odd"
                                        data-show="none" <?php echo( $conditional_item_key === 'date_odd' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'All odd dates', 'wpc-product-timer' ); ?>
								</option>
								<option value="time_range"
                                        data-show="time_range" <?php echo( $conditional_item_key === 'time_range' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Daily time range', 'wpc-product-timer' ); ?>
								</option>
                                <option value="time_before"
                                        data-show="time" <?php echo( $conditional_item_key === 'time_before' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Daily before time', 'wpc-product-timer' ); ?>
								</option>
								<option value="time_after"
                                        data-show="time" <?php echo( $conditional_item_key === 'time_after' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Daily after time', 'wpc-product-timer' ); ?>
								</option>
                                <option value="weekly_every"
                                        data-show="weekday" <?php echo( $conditional_item_key === 'weekly_every' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Weekly on every', 'wpc-product-timer' ); ?>
								</option>
                                <option value="week_no"
                                        data-show="weekno" <?php echo( $conditional_item_key === 'week_no' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'On week No.', 'wpc-product-timer' ); ?>
								</option>
                                <option value="every_day"
                                        data-show="none" <?php echo( $conditional_item_key === 'every_day' ? 'selected' : '' ); ?>>
									<?php esc_html_e( 'Everyday', 'wpc-product-timer' ); ?>
								</option>
							</select>
						</span>
                                <span class="woopt_hide woopt_show_if_date_time">
							<input value="<?php echo $conditional_item_val; ?>"
                                   class="woopt_date_time woopt_date_time_input woopt_picker"
                                   type="text"
                                   readonly="readonly" style="width: 300px"/>
						</span>
                                <span class="woopt_hide woopt_show_if_date">
							<input value="<?php echo $conditional_item_val; ?>"
                                   class="woopt_date woopt_date_input woopt_picker"
                                   type="text"
                                   readonly="readonly" style="width: 300px"/>
						</span>
                                <span class="woopt_hide woopt_show_if_date_range">
							<input value="<?php echo $conditional_item_val; ?>"
                                   class="woopt_date_range woopt_date_input woopt_picker"
                                   type="text"
                                   readonly="readonly" style="width: 300px"/>
						</span>
                                <span class="woopt_hide woopt_show_if_date_multi">
							<input value="<?php echo $conditional_item_val; ?>"
                                   class="woopt_date_multi woopt_date_input woopt_picker"
                                   type="text"
                                   readonly="readonly" style="width: 300px"/>
						</span>
                                <span class="woopt_hide woopt_show_if_time_range">
							<input value="<?php echo $conditional_item_time_from; ?>"
                                   class="woopt_time woopt_time_start woopt_time_input woopt_picker" type="text"
                                   readonly="readonly" style="width: 300px" placeholder="from"/>
							<input value="<?php echo $conditional_item_time_to; ?>"
                                   class="woopt_time woopt_time_end woopt_time_input woopt_picker"
                                   type="text"
                                   readonly="readonly"
                                   style="width: 300px" placeholder="to"/>
						</span>
                                <span class="woopt_hide woopt_show_if_time">
							<input value="<?php echo $conditional_item_val; ?>"
                                   class="woopt_time woopt_time_on woopt_time_input woopt_picker"
                                   type="text"
                                   readonly="readonly" style="width: 300px"/>
						</span>
                                <span class="woopt_hide woopt_show_if_weekday">
							<select class="woopt_weekday">
                                <option value="mon" <?php echo( $conditional_item_val === 'mon' ? 'selected' : '' ); ?>><?php esc_html_e( 'Monday', 'wpc-product-timer' ); ?></option>
                                <option value="tue" <?php echo( $conditional_item_val === 'tue' ? 'selected' : '' ); ?>><?php esc_html_e( 'Tuesday', 'wpc-product-timer' ); ?></option>
                                <option value="wed" <?php echo( $conditional_item_val === 'wed' ? 'selected' : '' ); ?>><?php esc_html_e( 'Wednesday', 'wpc-product-timer' ); ?></option>
                                <option value="thu" <?php echo( $conditional_item_val === 'thu' ? 'selected' : '' ); ?>><?php esc_html_e( 'Thursday', 'wpc-product-timer' ); ?></option>
                                <option value="fri" <?php echo( $conditional_item_val === 'fri' ? 'selected' : '' ); ?>><?php esc_html_e( 'Friday', 'wpc-product-timer' ); ?></option>
                                <option value="sat" <?php echo( $conditional_item_val === 'sat' ? 'selected' : '' ); ?>><?php esc_html_e( 'Saturday', 'wpc-product-timer' ); ?></option>
                                <option value="sun" <?php echo( $conditional_item_val === 'sun' ? 'selected' : '' ); ?>><?php esc_html_e( 'Sunday', 'wpc-product-timer' ); ?></option>
                            </select>
						</span>
                                <span class="woopt_hide woopt_show_if_weekno">
							<select class="woopt_weekno">
                                <?php
                                for ( $i = 1; $i < 54; $i ++ ) {
	                                echo '<option value="' . esc_attr( $i ) . '" ' . ( (int) $conditional_item_val === $i ? 'selected' : '' ) . '>' . $i . '</option>';
                                }
                                ?>
                            </select>
						</span>
                            </div>
							<?php
						}
					}
				}

				function woopt_add_conditional() {
					if ( ! isset( $_POST['woopt_nonce'] ) || ! wp_verify_nonce( $_POST['woopt_nonce'], 'woopt_nonce' ) ) {
						die( 'Permissions check failed' );
					}

					self::woopt_conditional();
					die();
				}

				function woopt_save_option_field( $post_id ) {
					if ( isset( $_POST['woopt_actions'] ) && is_array( $_POST['woopt_actions'] ) ) {
						update_post_meta( $post_id, 'woopt_actions', $_POST['woopt_actions'] );
					} else {
						delete_post_meta( $post_id, 'woopt_actions' );
					}
				}

				function woopt_product_columns( $columns ) {
					$columns['woopt'] = esc_html__( 'Timer', 'wpc-product-timer' );

					return $columns;
				}

				function woopt_product_posts_custom_column( $column, $postid ) {
					if ( $column == 'woopt' ) {
						echo '<div class="woopt-icons">';

						// global actions
						if ( is_array( self::$global_actions ) && ( count( self::$global_actions ) > 0 ) ) {
							$global  = false;
							$running = false;

							foreach ( self::$global_actions as $global_action ) {
								$action_data        = self::woopt_action_data( $global_action );
								$action_apply       = $action_data['apply'];
								$action_apply_val   = $action_data['apply_val'];
								$action_conditional = $action_data['conditional'];

								if ( self::woopt_check_apply( $postid, $action_apply, $action_apply_val ) ) {
									$global = true;

									if ( ! empty( $action_conditional ) && self::woopt_check_conditional( $action_conditional ) ) {
										$running = true;
									}
								}
							}

							if ( $global ) {
								if ( $running ) {
									echo '<span class="woopt-icon woopt-icon-global running"><span class="dashicons dashicons-admin-site"></span></span>';
								} else {
									echo '<span class="woopt-icon woopt-icon-global"><span class="dashicons dashicons-admin-site"></span></span>';
								}
							}
						}

						$actions = get_post_meta( $postid, 'woopt_actions', true );

						if ( is_array( $actions ) && ( count( $actions ) > 0 ) ) {
							$running = false;

							foreach ( $actions as $action ) {
								$action_data = self::woopt_action_data( $action );

								if ( ! empty( $action_data['conditional'] ) && self::woopt_check_conditional( $action_data['conditional'] ) ) {
									$running = true;
								}
							}

							if ( $running ) {
								echo '<span class="woopt-icon running"><span class="dashicons dashicons-clock"></span></span>';
							} else {
								echo '<span class="woopt-icon"><span class="dashicons dashicons-clock"></span></span>';
							}
						}

						echo '</div>';
					}
				}

				function woopt_add_export_column( $columns ) {
					$columns['woopt_actions'] = esc_html__( 'Timer', 'wpc-product-timer' );

					return $columns;
				}

				function woopt_add_export_data( $value, $product ) {
					$value = get_post_meta( $product->get_id(), 'woopt_actions', true );

					return base64_encode( serialize( $value ) );
				}

				function woopt_add_column_to_importer( $options ) {
					$options['woopt_actions']       = esc_html__( 'Timer', 'wpc-product-timer' );
					$options['woopt_unpurchasable'] = esc_html__( 'Unpurchasable', 'wpc-product-timer' );

					return $options;
				}

				function woopt_add_column_to_mapping_screen( $columns ) {
					$columns['Timer']               = 'woopt_actions';
					$columns['timer']               = 'woopt_actions';
					$columns['woopt actions']       = 'woopt_actions';
					$columns['Unpurchasable']       = 'woopt_unpurchasable';
					$columns['unpurchasable']       = 'woopt_unpurchasable';
					$columns['woopt unpurchasable'] = 'woopt_unpurchasable';

					return $columns;
				}

				function woopt_process_import( $object, $data ) {
					if ( ! empty( $data['woopt_unpurchasable'] ) ) {
						// date time format 03/05/2020 05:35 am
						$timer = 'set_unpurchasable||date_time_after>' . $data['woopt_unpurchasable'];
						$object->update_meta_data( 'woopt_actions', array( $timer ) );
					} elseif ( ! empty( $data['woopt_actions'] ) ) {
						$object->update_meta_data( 'woopt_actions', unserialize( base64_decode( $data['woopt_actions'] ) ) );
					}

					return $object;
				}
			}

			new WPCleverWoopt();
		}
	}
} else {
	add_action( 'admin_notices', 'woopt_notice_premium' );
}

if ( ! function_exists( 'woopt_notice_wc' ) ) {
	function woopt_notice_wc() {
		?>
        <div class="error">
            <p><strong>WPC Product Timer</strong> require WooCommerce version 3.0 or greater.</p>
        </div>
		<?php
	}
}

if ( ! function_exists( 'woopt_notice_premium' ) ) {
	function woopt_notice_premium() {
		?>
        <div class="error">
            <p>Seems you're using both free and premium version of <strong>WPC Product Timer</strong>. Please
                deactivate the free version when using the premium version.</p>
        </div>
		<?php
	}
}